<?php
// Cache utility for API responses
class ApiCache {
    private static $cacheDir = __DIR__ . '/../cache/';
    private static $defaultTtl = 180; // 3 minutes in seconds
    
    public static function init() {
        if (!file_exists(self::$cacheDir)) {
            mkdir(self::$cacheDir, 0755, true);
        }
    }
    
    public static function get($key) {
        self::init();
        $cacheFile = self::$cacheDir . md5($key) . '.json';
        
        if (!file_exists($cacheFile)) {
            return null;
        }
        
        $data = json_decode(file_get_contents($cacheFile), true);
        
        if (!$data || !isset($data['expires']) || time() > $data['expires']) {
            // Cache expired or invalid
            unlink($cacheFile);
            return null;
        }
        
        return $data['data'];
    }
    
    public static function set($key, $data, $ttl = null) {
        self::init();
        $cacheFile = self::$cacheDir . md5($key) . '.json';
        
        $cacheData = [
            'data' => $data,
            'expires' => time() + ($ttl ?? self::$defaultTtl),
            'created' => time()
        ];
        
        file_put_contents($cacheFile, json_encode($cacheData));
    }
    
    public static function delete($key) {
        self::init();
        $cacheFile = self::$cacheDir . md5($key) . '.json';
        
        if (file_exists($cacheFile)) {
            unlink($cacheFile);
        }
    }
    
    public static function clear() {
        self::init();
        $files = glob(self::$cacheDir . '*.json');
        foreach ($files as $file) {
            unlink($file);
        }
    }
    
    public static function getInfo($key) {
        self::init();
        $cacheFile = self::$cacheDir . md5($key) . '.json';
        
        if (!file_exists($cacheFile)) {
            return null;
        }
        
        $data = json_decode(file_get_contents($cacheFile), true);
        
        if (!$data) {
            return null;
        }
        
        return [
            'created' => $data['created'],
            'expires' => $data['expires'],
            'is_expired' => time() > $data['expires'],
            'ttl_remaining' => max(0, $data['expires'] - time())
        ];
    }
}
?>
