<?php

require_once __DIR__ . '/../utils/ApiHelper.php';

class MarketDataService {
    private $finnhubKey;

    public function __construct() {
        // ideally load from env
        $this->finnhubKey = getenv('FINNHUB_API_KEY') ?: 'demo'; 
    }

    public function getStockPrice($symbol) {
        $url = "https://finnhub.io/api/v1/quote?symbol={$symbol}&token={$this->finnhubKey}";
        $data = ApiHelper::get($url);

        if ($data && isset($data['c'])) {
            return [
                'price' => $data['c'],
                'change' => $data['d'],
                'percent' => $data['dp']
            ];
        }

        // Fallback Mock Data
        return $this->getMockStockData($symbol);
    }

    public function getCryptoPrice($id) {
        // CoinGecko Free API - No Key Needed
        $url = "https://api.coingecko.com/api/v3/simple/price?ids={$id}&vs_currencies=usd&include_24hr_change=true";
        $data = ApiHelper::get($url);

        if ($data && isset($data[$id])) {
            return [
                'price' => $data[$id]['usd'],
                'change' => $data[$id]['usd_24h_change'] ?? 0
            ];
        }
        
        return $this->getMockCryptoData($id);
    }
    
    public function getCommodityPrice($asset) {
        // Map commodities to proxy assets (e.g. Gold -> PAXG)
        $map = [
            'gold' => 'pax-gold',
            'silver' => 'kinesis-silver', // approximated
            'oil' => 'ether-oil' // approximated or specific proxy
        ];
        
        $id = $map[$asset] ?? $asset;
        return $this->getCryptoPrice($id);
    }

    private function getMockStockData($symbol) {
        // Realistic random price generation based on symbol hash
        $hash = crc32($symbol);
        $base = ($hash % 1000) / 10 + 50; // Random price 50-150
        $variation = (rand(-50, 50) / 100);
        
        return [
            'price' => $base,
            'change' => $variation,
            'percent' => ($variation / $base) * 100
        ];
    }

    private function getMockCryptoData($id) {
         $hash = crc32($id);
         $base = ($hash % 50000) + 100;
         return [
             'price' => $base,
             'change' => rand(-5, 5)
         ];
    }
}
