<?php
require_once __DIR__ . '/vendor/autoload.php';

$dotenv = Dotenv\Dotenv::createImmutable(__DIR__);
$dotenv->safeLoad();

// --- CORS HEADERS ---
// CORS Headers
$allowed_origins = [
    $_ENV['FRONTEND_URL'] ?? 'http://localhost:5173',
    'https://ofidel.org',
    'http://localhost:3000'
];

$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
if (in_array($origin, $allowed_origins)) {
    header("Access-Control-Allow-Origin: $origin");
} else {
    // Default to the first allowed origin if no match (or for tools)
    // header("Access-Control-Allow-Origin: " . $allowed_origins[0]);
    // Better to not send the header if origin doesn't match, or send * if strictly development
    // For security, we should be strict.
    if (empty($origin)) {
        // Allow non-browser requests (like curl) or handle as needed
    }
}

header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Access-Control-Allow-Credentials: true');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/handlers/authHandler.php';
require_once __DIR__ . '/handlers/transactionHandler.php';
require_once __DIR__ . '/handlers/managerHandler.php';
require_once __DIR__ . '/handlers/paymentHandler.php';
require_once __DIR__ . '/handlers/settingsHandler.php';
require_once __DIR__ . '/handlers/investmentHandler.php';
require_once __DIR__ . '/handlers/messageHandler.php';
require_once __DIR__ . '/handlers/jointAccountHandler.php';
require_once __DIR__ . '/handlers/adminHandler.php';
require_once __DIR__ . '/handlers/paymentMethodsHandler.php';
require_once __DIR__ . '/handlers/demoFundHandler.php';
require_once __DIR__ . '/handlers/adminAutoInvestHandler.php';

$routes = [
    // Auth endpoints
    ['POST',   '#^/api/auth/login$#',    'handleAuthRequest'],
    ['POST',   '#^/api/auth/register$#', 'handleAuthRequest'],
    ['POST',   '#^/api/auth/signup-with-invitation$#', 'handleAuthRequest'],
    ['POST',   '#^/api/auth/refresh$#',  'handleAuthRequest'],
    ['POST',   '#^/api/auth/logout$#',   'handleAuthRequest'],
    ['GET',    '#^/api/user/profile$#',  'handleAuthRequest'],
    ['GET',    '#^/api/user/balance$#',  'handlePaymentRequest'],

    // Manager endpoints
    ['GET',    '#^/api/manager/profile$#',            'handleManagerRequest'],
    ['PUT',    '#^/api/manager/profile$#',            'handleManagerRequest'],
    ['GET',    '#^/api/manager/permissions$#',        'handleManagerRequest'],
    ['PUT',    '#^/api/manager/permissions$#',        'handleManagerRequest'],
    ['GET',    '#^/api/manager/campaigns$#',          'handleManagerRequest'],
    ['POST',   '#^/api/manager/campaigns$#',          'handleManagerRequest'],
    ['PUT',    '#^/api/manager/campaigns/\\d+/status$#', 'handleManagerRequest'],
    ['GET',    '#^/api/manager/investors$#',          'handleManagerRequest'],
    ['POST',   '#^/api/manager/investors/bulk-action$#', 'handleManagerRequest'],
    ['GET',    '#^/api/manager/approvals$#',          'handleManagerRequest'],
    ['POST',   '#^/api/manager/approvals$#',          'handleManagerRequest'],
    ['GET',    '#^/api/manager/users$#',              'handleManagerRequest'],
    ['GET',    '#^/api/manager/stats$#',              'handleManagerRequest'],
    ['GET',    '#^/api/manager/notifications$#',      'handleManagerRequest'],
    ['GET',    '#^/api/manager/investments$#',        'handleManagerRequest'],
    ['GET',    '#^/api/manager/users/\d+/investments$#', 'handleManagerRequest'],
    ['POST',   '#^/api/manager/messages$#',           'handleManagerRequest'],
    ['GET',    '#^/api/manager/messages/\\d+$#',      'handleManagerRequest'],
    ['GET',    '#^/api/manager/activity-log$#',       'handleManagerRequest'],
    ['POST',   '#^/api/manager/users/\\d+/(pause-demo|resume-demo)$#', 'handleManagerRequest'],

    // Payment endpoints
    ['GET',    '#^/api/payments$#',                  'handlePaymentRequest'],
    ['POST',   '#^/api/payments$#',                   'handlePaymentRequest'],
    ['PATCH',  '#^/api/payments/\\d+/status$#',       'handlePaymentRequest'],
    ['GET',    '#^/api/payments/status/\\w+$#',       'handlePaymentRequest'],
    ['GET',    '#^/api/payments/balance$#',           'handlePaymentRequest'],
    ['POST',   '#^/api/payments/balance/validate$#',  'handlePaymentRequest'],
    ['POST',   '#^/api/payments/balance/sync$#',      'handlePaymentRequest'],
    ['POST',   '#^/api/payments/\\d+/refund$#',       'handlePaymentRequest'],
    ['POST',   '#^/api/payments/provider$#',          'handlePaymentRequest'],

    // Transaction endpoints (kept generic)
    ['GET',    '#^/api/transactions$#', 'handleTransactionRequest'],

    // Investment endpoints
    ['POST',   '#^/api/investments$#', 'handleInvestmentRequest'],
    ['GET',    '#^/api/investments/user$#', 'handleInvestmentRequest'],
    ['GET',    '#^/api/investments/(\d+)$#', 'handleInvestmentRequest'],
    ['POST',   '#^/api/investments/update$#', 'handleInvestmentRequest'],
    ['POST',   '#^/api/investments/status$#', 'handleInvestmentRequest'],
    ['POST',   '#^/api/investments/pnl$#', 'handleInvestmentRequest'],
    ['GET',    '#^/api/investments/(\d+)/pnl$#', 'handleInvestmentRequest'],
    ['GET',    '#^/api/investments/portfolio$#', 'handleInvestmentRequest'],

    // Message endpoints
    ['POST',   '#^/api/messages$#', 'handleMessageRequest'],
    ['GET',    '#^/api/messages/\\d+$#', 'handleMessageRequest'],
    ['PUT',    '#^/api/messages/\\d+/read$#', 'handleMessageRequest'],
    ['GET',    '#^/api/messages/(\d+)$#', 'handleMessageRequest'],
    ['POST',   '#^/api/messages/send$#', 'handleMessageRequest'],
    ['GET',    '#^/api/messages/conversations$#', 'handleMessageRequest'],

    // Joint Account endpoints
    ['GET',    '#^/api/joint-accounts$#', 'handleJointAccountRequest'],
    ['POST',   '#^/api/joint-accounts/invite$#', 'handleJointAccountRequest'],
    ['POST',   '#^/api/joint-accounts/validate-invitation$#', 'handleJointAccountRequest'],
    ['POST',   '#^/api/joint-accounts/accept$#', 'handleJointAccountRequest'],
    ['GET',    '#^/api/joint-accounts/\\d+$#', 'handleJointAccountRequest'],
    ['POST',   '#^/api/joint-accounts/transactions$#', 'handleJointAccountRequest'],
    ['POST',   '#^/api/joint-accounts/transactions/approve$#', 'handleJointAccountRequest'],
    ['GET',    '#^/api/joint-accounts/\\d+/pending-transactions$#', 'handleJointAccountRequest'],
    ['DELETE', '#^/api/joint-accounts/\\d+/members/\\d+$#', 'handleJointAccountRequest'],
    ['POST',   '#^/api/joint-accounts/\\d+/trade$#', 'handleJointAccountRequest'],
    ['GET',    '#^/api/joint-accounts/\\d+/positions$#', 'handleJointAccountRequest'],
    ['GET',    '#^/api/joint-accounts/\\d+/orders$#', 'handleJointAccountRequest'],
    ['GET',    '#^/api/joint-accounts/\\d+/performance$#', 'handleJointAccountRequest'],

    // User demo status and withdrawal/cancellation endpoints
    ['GET',    '#^/api/user/demo-status$#',           'handleAuthRequest'],
    ['PUT',    '#^/api/user/demo-notified$#',         'handleAuthRequest'],
    ['POST',   '#^/api/investments/\\d+/withdraw$#',  'handleAuthRequest'],
    ['POST',   '#^/api/investments/\\d+/cancel$#',    'handleAuthRequest'],
    
    // Admin endpoints
    ['GET',    '#^/api/admin/dashboard$#',            'handleAdminRequest'],
    ['GET',    '#^/api/admin/users$#',                'handleAdminRequest'],
    ['GET',    '#^/api/admin/users/\\d+$#',           'handleAdminRequest'],
    ['POST',   '#^/api/admin/users/\\d+/suspend$#',   'handleAdminRequest'],
    ['POST',   '#^/api/admin/users/\\d+/activate$#',  'handleAdminRequest'],
    ['POST',   '#^/api/admin/users/\\d+/delete$#',    'handleAdminRequest'],
    ['GET',    '#^/api/admin/analytics$#',            'handleAdminRequest'],
    ['GET',    '#^/api/admin/alerts$#',               'handleAdminRequest'],
    ['GET',    '#^/api/admin/recent-activity$#',      'handleAdminRequest'],
    ['GET',    '#^/api/admin/payment-requests$#',         'handleAdminRequest'],
    ['GET',    '#^/api/admin/payment-details$#',          'handleAdminRequest'],
    ['POST',   '#^/api/admin/payment-details$#',          'handleAdminRequest'],
    ['GET',    '#^/api/admin/payment-requests/\\d+$#',    'handleAdminRequest'],
    ['PUT',    '#^/api/admin/payment-requests/\\d+$#',    'handleAdminRequest'],
    ['POST',   '#^/api/admin/approve-deposit/\\d+$#',     'handleAdminRequest'],
    ['GET',    '#^/api/admin/payment-methods$#',          'handleAdminRequest'],
    ['POST',   '#^/api/admin/payment-methods$#',          'handleAdminRequest'],
    ['PUT',    '#^/api/admin/payment-methods/\\d+$#',     'handleAdminRequest'],
    ['DELETE', '#^/api/admin/payment-methods/\\d+$#',     'handleAdminRequest'],
    ['POST',   '#^/api/admin/send-email$#',              'handleAdminRequest'],
    ['GET',    '#^/api/payment-methods$#',                'handlePaymentMethodsRequest'],
    ['GET',    '#^/api/payment-methods/\\d+$#',           'handlePaymentMethodsRequest'],
    ['GET',    '#^/api/payment-requests$#',               'handlePaymentMethodsRequest'],
    ['POST',   '#^/api/payment-requests$#',               'handlePaymentMethodsRequest'],
    ['PUT',    '#^/api/payment-requests/\\d+$#',          'handlePaymentMethodsRequest'],
    ['POST',   '#^/api/payment-requests/\\d+/proof$#',    'handlePaymentMethodsRequest'],
    
    // Demo fund application routes
    ['GET',    '#^/api/demo-fund-applications$#',                'handleDemoFundRequest'],
    ['POST',   '#^/api/demo-fund-applications$#',               'handleDemoFundRequest'],
    ['GET',    '#^/api/admin/demo-fund-applications$#',         'handleAdminDemoFundRequest'],
    ['PUT',    '#^/api/admin/demo-fund-applications/\d+/approve$#', 'handleAdminDemoFundRequest'],
    ['PUT',    '#^/api/admin/demo-fund-applications/\d+/reject$#',  'handleAdminDemoFundRequest'],
    
    // Admin Auto-Invest management routes
    ['GET',    '#^/api/admin/auto-invest/plans$#',        'handleAdminAutoInvest'],
    ['POST',   '#^/api/admin/auto-invest/plans$#',        'handleAdminAutoInvest'],
    ['PUT',    '#^/api/admin/auto-invest/plans/\d+$#',   'handleAdminAutoInvest'],
    ['DELETE', '#^/api/admin/auto-invest/plans/\d+$#',   'handleAdminAutoInvest'],
];

// Route the request
$method = $_SERVER['REQUEST_METHOD'];
$path = $_SERVER['REQUEST_URI'];
$path = preg_replace('#^/backend#', '', $path); // Strip /backend from path for routing
$path = explode('?', $path, 2)[0]; // Strip query string for routing
$input = json_decode(file_get_contents('php://input'), true) ?? [];
$headers = getallheaders();

// Investment Options endpoints
if (preg_match('#^/api/investment-options#', $path)) {
    include __DIR__ . '/api/investment-options.php';
    exit();
}

// Invest Opportunity endpoint
if ($path === '/api/invest-opportunity') {
    include __DIR__ . '/api/invest-opportunity.php';
    exit();
}

// Special Plans endpoints (Auto-Invest)
if (preg_match('#^/api/auto-invest#', $path) || preg_match('#^/api/special-plans#', $path)) {
    require_once __DIR__ . '/controllers/AutoInvestController.php';
    $controller = new AutoInvestController();
    
    // Extract User ID from token for secure methods
    $userId = null;
    if (isset($headers['Authorization'])) {
        require_once __DIR__ . '/utils/jwt.php';
        $token = str_replace('Bearer ', '', $headers['Authorization']);
        $payload = verify_jwt($token);
        if ($payload) $userId = $payload['sub'];
    }

    if ($method === 'GET' && $path === '/api/auto-invest/plans') {
        $controller->index();
        exit();
    }
    if ($method === 'GET' && $path === '/api/auto-invest/my-investments') {
        if (!$userId) { http_response_code(401); echo json_encode(['error'=>'Unauthorized']); exit; }
        $controller->getUserInvestments($userId);
        exit();
    }
    if ($method === 'POST' && $path === '/api/auto-invest/invest') {
        if (!$userId) { http_response_code(401); echo json_encode(['error'=>'Unauthorized']); exit; }
        $controller->invest($userId);
        exit();
    }
    
    // Legacy support for special-plans includes
    if (preg_match('#^/api/special-plans#', $path)) {
         include __DIR__ . '/api/special-plans.php';
         exit();
    }
}

// Wallet endpoints
if (preg_match('#^/api/wallet#', $path)) {
    require_once __DIR__ . '/controllers/WalletController.php';
    $controller = new WalletController();
    
    // Auth Check
    $userId = null;
    if (isset($headers['Authorization'])) {
        require_once __DIR__ . '/utils/jwt.php';
        $token = str_replace('Bearer ', '', $headers['Authorization']);
        $payload = verify_jwt($token);
        if ($payload) $userId = $payload['sub'];
    }

    if ($method === 'POST' && $path === '/api/wallet/generate') {
         if (!$userId) { http_response_code(401); echo json_encode(['error'=>'Unauthorized']); exit; }
         $_POST['user_id'] = $userId;
         $controller->generate();
         exit();
    }
    if ($method === 'GET' && $path === '/api/wallet/private-key') {
         // Admin check inside controller or here
         $controller->getPrivateKey();
         exit();
    }
}


// Portfolio Summary
if ($method === 'GET' && $path === '/api/portfolio/summary') {
    require_once __DIR__ . '/controllers/PortfolioController.php';
    $controller = new PortfolioController();
    $userId = null;
    if (isset($headers['Authorization'])) {
        require_once __DIR__ . '/utils/jwt.php';
        $token = str_replace('Bearer ', '', $headers['Authorization']);
        $payload = verify_jwt($token);
        if ($payload) $userId = $payload['sub'];
    }
    if (!$userId) { http_response_code(401); echo json_encode(['error'=>'Unauthorized']); exit; }
    $controller->getSummary($userId);
    exit();
}

$matched = false;
foreach ($routes as $route) {
    list($routeMethod, $routePattern, $handler) = $route;
    if ($method === $routeMethod && preg_match($routePattern, $path)) {
        $result = $handler($method, $path, $input, $headers);
        header('Content-Type: application/json');
        echo json_encode($result);
        exit();
    }
}
// If no route matched
header('Content-Type: application/json');
echo json_encode(['error' => 'Not found', 'code' => 404, 'debug' => ['method' => $method, 'path' => $path]]);
exit(); 