<?php
require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../utils/jwt.php';

function handlePaymentMethodsRequest($method, $path, $input, $headers) {
    $pdo = getPDO();
    
    // Verify JWT token
    $user = verify_jwt_from_headers($headers);
    if (!$user) {
        http_response_code(401);
        return ['error' => 'Unauthorized', 'code' => 401];
    }

    switch ($method) {
        case 'GET':
            if (preg_match('#^/api/payment-methods$#', $path)) {
                return getPaymentMethods($pdo, $user);
            } elseif (preg_match('#^/api/payment-methods/(\d+)$#', $path, $matches)) {
                return getPaymentMethod($pdo, $matches[1], $user);
            } elseif (preg_match('#^/api/payment-requests$#', $path)) {
                $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
                $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
                return getUserPaymentRequests($pdo, $user, $page, $limit);
            }
            break;
            
        case 'POST':
            if (preg_match('#^/api/payment-requests$#', $path)) {
                return createPaymentRequest($pdo, $user, $input);
            } elseif (preg_match('#^/api/payment-requests/(\d+)/proof$#', $path, $matches)) {
                return uploadProof($pdo, $matches[1], $user, $input);
            }
            break;
            
        case 'PUT':
            if (preg_match('#^/api/payment-requests/(\d+)$#', $path, $matches)) {
                return updateUserPaymentRequest($pdo, $matches[1], $user, $input);
            }
            break;
    }
    
    return ['error' => 'Endpoint not found', 'code' => 404];
}

function getPaymentMethods($pdo, $user) {
    try {
        $stmt = $pdo->prepare('
            SELECT id, name, type, description, instructions, account_details, 
                   is_active, is_default, min_amount, max_amount, processing_time, fees
            FROM payment_methods 
            WHERE is_active = TRUE 
            ORDER BY is_default DESC, name ASC
        ');
        $stmt->execute();
        $methods = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Decode JSON account_details
        foreach ($methods as &$method) {
            $method['account_details'] = json_decode($method['account_details'], true);
        }
        
        return ['success' => true, 'payment_methods' => $methods];
    } catch (Exception $e) {
        // If payment_methods table doesn't exist, return default methods
        $defaultMethods = [
            [
                'id' => 1,
                'name' => 'Wire Transfer',
                'type' => 'wire_transfer',
                'description' => 'International bank wire transfers via SWIFT',
                'instructions' => 'Please include your username as reference when making the transfer.',
                'account_details' => [
                    'bank_name' => 'Bank of America',
                    'account_number' => '1234567890',
                    'routing_number' => '021000021',
                    'account_holder' => 'Wealth Crypto Nexus LLC'
                ],
                'is_active' => true,
                'is_default' => true,
                'min_amount' => 100,
                'max_amount' => 100000,
                'processing_time' => '1-3 business days',
                'fees' => '$15-25'
            ],
            [
                'id' => 2,
                'name' => 'Cryptocurrency',
                'type' => 'cryptocurrency',
                'description' => 'Bitcoin, Ethereum, USDC payments',
                'instructions' => 'Send to the provided wallet address with reference number.',
                'account_details' => [
                    'wallet_address' => 'bc1qxy2kgdygjrsqtzq2n0yrf2493p83kkfjhx0wlh',
                    'network' => 'Bitcoin'
                ],
                'is_active' => true,
                'is_default' => false,
                'min_amount' => 10,
                'max_amount' => 100000,
                'processing_time' => '10-60 minutes',
                'fees' => 'Network fees'
            ]
        ];
        
        return ['success' => true, 'payment_methods' => $defaultMethods];
    }
}

function getPaymentMethod($pdo, $methodId, $user) {
    try {
        $stmt = $pdo->prepare('
            SELECT id, name, type, description, instructions, account_details, 
                   is_active, is_default, min_amount, max_amount, processing_time, fees
            FROM payment_methods 
            WHERE id = ? AND is_active = TRUE
        ');
        $stmt->execute([$methodId]);
        $method = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$method) {
            return ['error' => 'Payment method not found', 'code' => 404];
        }
        
        $method['account_details'] = json_decode($method['account_details'], true);
        
        return ['success' => true, 'payment_method' => $method];
    } catch (Exception $e) {
        return ['error' => 'Failed to fetch payment method', 'code' => 500];
    }
}

function createPaymentRequest($pdo, $user, $input) {
    try {
        $data = $input;
        
        if (!isset($data['payment_method_id']) || !isset($data['amount']) || !isset($data['type'])) {
            http_response_code(400);
            return json_encode(['error' => 'Missing required fields']);
        }
        
        // Validate payment method exists and is active
        $stmt = $pdo->prepare('SELECT * FROM payment_methods WHERE id = ? AND is_active = TRUE');
        $stmt->execute([$data['payment_method_id']]);
        $paymentMethod = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$paymentMethod) {
            http_response_code(400);
            return json_encode(['error' => 'Invalid payment method']);
        }
        
        // Validate amount
        $amount = floatval($data['amount']);
        if ($amount < $paymentMethod['min_amount'] || $amount > $paymentMethod['max_amount']) {
            http_response_code(400);
            return json_encode(['error' => 'Amount is outside allowed range']);
        }
        
        // Generate reference number
        $referenceNumber = 'PAY-' . strtoupper(substr(md5(uniqid()), 0, 8)) . '-' . time();
        
        // Create payment request using existing table structure (temporary fix)
        $stmt = $pdo->prepare('
            INSERT INTO payment_requests (user_id, type, amount, status, payment_details, admin_notes)
            VALUES (?, ?, ?, ?, ?, ?)
        ');
        $stmt->execute([
            $user['sub'],
            $data['type'],
            $amount,
            'pending',
            json_encode([
                'payment_method_id' => $data['payment_method_id'],
                'currency' => $data['currency'] ?? 'USD',
                'reference_number' => $referenceNumber,
                'notes' => $data['notes'] ?? '',
                'session_id' => $data['session_id'] ?? null
            ]),
            ''
        ]);
        
        $requestId = $pdo->lastInsertId();
        
        // Get the created request
        $stmt = $pdo->prepare('SELECT * FROM payment_requests WHERE id = ?');
        $stmt->execute([$requestId]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Decode payment_details for frontend
        $request['payment_details'] = json_decode($request['payment_details'], true);
        
        return [
            'success' => true, 
            'message' => 'Payment request created successfully',
            'payment_request' => $request
        ];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to create payment request: ' . $e->getMessage(), 'code' => 500];
    }
}

function updateUserPaymentRequest($pdo, $requestId, $user, $input) {
    try {
        $data = $input;
        
        // Verify the request belongs to the user
        $stmt = $pdo->prepare('SELECT * FROM payment_requests WHERE id = ? AND user_id = ?');
        $stmt->execute([$requestId, $user['sub']]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            http_response_code(404);
            return json_encode(['error' => 'Payment request not found']);
        }
        
        // Update payment_details with new information
        $currentDetails = json_decode($request['payment_details'], true) ?? [];
        $updatedDetails = array_merge($currentDetails, [
            'notes' => $data['notes'] ?? $currentDetails['notes'],
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        $stmt = $pdo->prepare('UPDATE payment_requests SET payment_details = ? WHERE id = ? AND user_id = ?');
        $stmt->execute([json_encode($updatedDetails), $requestId, $user['sub']]);
        
        return ['success' => true, 'message' => 'Payment request updated successfully'];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to update payment request: ' . $e->getMessage(), 'code' => 500];
    }
}

function getUserPaymentRequests($pdo, $user, $page = 1, $limit = 10) {
    try {
        $offset = ($page - 1) * $limit;
        
        // Get total count
        $countStmt = $pdo->prepare('SELECT COUNT(*) as total FROM payment_requests WHERE user_id = ?');
        $countStmt->execute([$user['sub']]);
        $totalCount = $countStmt->fetch()['total'];
        
        // Get paginated results
        $stmt = $pdo->prepare('
            SELECT 
                pr.*,
                pm.name as payment_method_name,
                pm.type as payment_method_type
            FROM payment_requests pr
            LEFT JOIN payment_methods pm ON JSON_EXTRACT(pr.payment_details, "$.payment_method_id") = pm.id
            WHERE pr.user_id = ?
            ORDER BY pr.created_at DESC
            LIMIT ? OFFSET ?
        ');
        $stmt->bindValue(1, $user['sub'], PDO::PARAM_STR);
        $stmt->bindValue(2, (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(3, (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        $requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Decode payment_details for each request
        foreach ($requests as &$request) {
            if ($request['payment_details']) {
                $details = json_decode($request['payment_details'], true);
                $request['currency'] = $details['currency'] ?? 'USD';
                $request['reference_number'] = $details['reference_number'] ?? '';
                $request['user_notes'] = $details['notes'] ?? '';
            }
        }
        
        return [
            'success' => true, 
            'payment_requests' => $requests,
            'pagination' => [
                'current_page' => $page,
                'per_page' => $limit,
                'total' => $totalCount,
                'total_pages' => ceil($totalCount / $limit)
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Failed to fetch payment requests: ' . $e->getMessage(), 'code' => 500];
    }
}

function uploadProof($pdo, $requestId, $user, $input) {
    try {
        // Verify the request belongs to the user
        $stmt = $pdo->prepare('SELECT * FROM payment_requests WHERE id = ? AND user_id = ?');
        $stmt->execute([$requestId, $user['sub']]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            return ['error' => 'Payment request not found', 'code' => 404];
        }
        
        // Handle file upload
        if (!isset($_FILES['proof_file']) || $_FILES['proof_file']['error'] !== UPLOAD_ERR_OK) {
            return ['error' => 'Proof file is required and must be uploaded successfully', 'code' => 400];
        }
        
        $file = $_FILES['proof_file'];
        
        // Validate file type
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'application/pdf'];
        if (!in_array($file['type'], $allowedTypes)) {
            return ['error' => 'Invalid file type. Only images (JPEG, PNG, GIF) and PDF files are allowed.', 'code' => 400];
        }
        
        // Validate file size (max 10MB)
        if ($file['size'] > 10 * 1024 * 1024) {
            return ['error' => 'File size too large. Maximum size is 10MB.', 'code' => 400];
        }
        
        // Create uploads directory if it doesn't exist
        $uploadDir = __DIR__ . '/../../uploads/proofs/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        // Generate unique filename
        $fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $fileName = 'proof_' . $requestId . '_' . time() . '.' . $fileExtension;
        $filePath = $uploadDir . $fileName;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $filePath)) {
            return ['error' => 'Failed to save uploaded file', 'code' => 500];
        }
        
        // Update payment_details with proof file information
        $currentDetails = json_decode($request['payment_details'], true) ?? [];
        $updatedDetails = array_merge($currentDetails, [
            'proof_file' => $fileName,
            'proof_uploaded_at' => date('Y-m-d H:i:s')
        ]);
        
        // Update the request with proof file
        $stmt = $pdo->prepare('UPDATE payment_requests SET payment_details = ? WHERE id = ? AND user_id = ?');
        $stmt->execute([json_encode($updatedDetails), $requestId, $user['sub']]);
        
        return [
            'success' => true, 
            'message' => 'Proof uploaded successfully. Your payment is now being reviewed.',
            'proof_file' => $fileName
        ];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to upload proof: ' . $e->getMessage(), 'code' => 500];
    }
}
?>
