<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE, PATCH');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../controllers/ManagerController.php';
require_once __DIR__ . '/../utils/jwt.php';

function handleManagerRequest($method, $path, $input, $headers) {
    $manager = new ManagerController();

    if (!isset($headers['Authorization'])) return ['error' => 'Missing token', 'code' => 401];
    $token = str_replace('Bearer ', '', $headers['Authorization']);
    $payload = verify_jwt($token);
    if (!$payload) return ['error' => 'Invalid or expired token', 'code' => 401];
    $user_id = $payload['sub'];

    // Role check: Only allow users with role 'manager'
    $pdo = $manager->getPDO();
    $stmt = $pdo->prepare('SELECT role FROM users WHERE id = ?');
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user || $user['role'] !== 'manager') {
        return ['error' => 'Access denied. Only managers can access this endpoint.', 'code' => 403];
    }

    // Get manager ID from managers table
    $stmt = $pdo->prepare('SELECT id FROM managers WHERE user_id = ?');
    $stmt->execute([$user_id]);
    $managerData = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$managerData) {
        return ['error' => 'Manager profile not found', 'code' => 404];
    }
    $manager_id = $managerData['id'];

    if ($method === 'GET' && preg_match('#/api/manager/profile$#', $path)) {
        return ['profile' => $manager->getProfile($user_id)];
    }
    if ($method === 'GET' && preg_match('#/api/manager/approvals$#', $path)) {
        return ['approvals' => $manager->getApprovals($user_id)];
    }
    if ($method === 'POST' && preg_match('#/api/manager/approvals$#', $path)) {
        if (!isset($input['user_id'], $input['status'])) return ['error' => 'user_id and status required', 'code' => 400];
        return $manager->approveUser($user_id, $input['user_id'], $input['status']);
    }
    if ($method === 'GET' && preg_match('#/api/manager/users$#', $path)) {
        return ['users' => $manager->getManagedUsers($manager_id)];
    }
    if ($method === 'GET' && preg_match('#/api/manager/stats$#', $path)) {
        return $manager->getManagerStats($manager_id);
    }
    if ($method === 'GET' && preg_match('#^/api/manager/investments$#', $path)) {
        return $manager->getManagerInvestments($user_id);
    }
    if ($method === 'GET' && preg_match('#^/api/manager/users/(\d+)/investments$#', $path, $matches)) {
        return $manager->getUserInvestmentsForManager($user_id, intval($matches[1]));
    }
    if ($method === 'GET' && preg_match('#/api/manager/notifications$#', $path)) {
        return ['notifications' => $manager->getNotifications($user_id)];
    }
    if ($method === 'POST' && preg_match('#/api/manager/messages$#', $path)) {
        if (!isset($input['to'], $input['content'])) return ['error' => 'to and content required', 'code' => 400];
        return $manager->sendMessage($user_id, $input['to'], $input['content']);
    }
    if ($method === 'GET' && preg_match('#/api/manager/messages/(\d+)$#', $path, $matches)) {
        return ['messages' => $manager->getMessages($user_id, $matches[1])];
    }
    if ($method === 'GET' && preg_match('#/api/manager/activity-log$#', $path)) {
        return ['activity_log' => $manager->getActivityLog($user_id)];
    }
    if ($method === 'PUT' && preg_match('#/api/manager/profile$#', $path)) {
        return $manager->updateProfile($manager_id, $input);
    }
    if ($method === 'PUT' && preg_match('#/api/manager/permissions$#', $path)) {
        return $manager->updatePermissions($manager_id, $input);
    }

    // Manager can pause/resume user's demo transactions
    if ($method === 'POST' && preg_match('#^/api/manager/users/(\d+)/(pause-demo|resume-demo)$#', $path, $matches)) {
        $target_user_id = intval($matches[1]);
        $action = $matches[2];
        $paused = $action === 'pause-demo';
        $reason = $input['reason'] ?? null;
        $expires_at = $input['expires_at'] ?? null; // optional
        return $manager->setUserDemoPaused($user_id, $target_user_id, $paused, $reason, $expires_at);
    }
    if ($method === 'PUT' && preg_match('#/api/manager/campaigns/(\d+)/status$#', $path, $matches)) {
        if (!isset($input['status']) || !in_array($input['status'], ['active', 'paused', 'completed', 'cancelled'])) return ['error' => 'Invalid status', 'code' => 400];
        return $manager->updateCampaignStatus($manager_id, $matches[1], $input['status']);
    }
    if ($method === 'GET' && preg_match('#/api/manager/campaigns$#', $path)) {
        return $manager->getCampaigns($manager_id);
    }
    if ($method === 'GET' && preg_match('#/api/manager/investors$#', $path)) {
        $page = $_GET['page'] ?? 1;
        $limit = $_GET['limit'] ?? 10;
        $search = $_GET['search'] ?? '';
        return $manager->getInvestors($manager_id, $page, $limit, $search);
    }
    if ($method === 'POST' && preg_match('#/api/manager/campaigns$#', $path)) {
        $data = json_decode(file_get_contents('php://input'), true);
        return $manager->createCampaign($manager_id, $data);
    }
    if ($method === 'POST' && preg_match('#/api/manager/investors/bulk-action$#', $path)) {
        $data = json_decode(file_get_contents('php://input'), true);
        return $manager->handleBulkInvestorAction($manager_id, $data);
    }
    return ['error' => 'Not found', 'code' => 404];
}

function handleBrokerRequest($method, $path, $input, $headers) {
    $manager = new ManagerController();
    // List all brokers (public)
    if ($method === 'GET' && preg_match('#^/api/brokers$#', $path)) {
        return ['brokers' => $manager->getAllBrokers()];
    }
    // Broker detail (public)
    if ($method === 'GET' && preg_match('#^/api/brokers/(\d+)$#', $path, $matches)) {
        $user_id = null;
        if (isset($headers['Authorization'])) {
            require_once __DIR__ . '/../utils/jwt.php';
            $token = str_replace('Bearer ', '', $headers['Authorization']);
            $payload = verify_jwt($token);
            if ($payload) {
                $user_id = $payload['sub'];
            }
        }
        return $manager->getBrokerDetail($matches[1], $user_id);
    }
    // Apply to be a broker
    if ($method === 'POST' && preg_match('#^/api/brokers/apply$#', $path)) {
        // Extract user_id from JWT
        if (!isset($headers['Authorization'])) return ['error' => 'Missing token', 'code' => 401];
        require_once __DIR__ . '/../utils/jwt.php';
        $token = str_replace('Bearer ', '', $headers['Authorization']);
        $payload = verify_jwt($token);
        if (!$payload) return ['error' => 'Invalid or expired token', 'code' => 401];
        $user_id = $payload['sub'];

        // Support both JSON and multipart/form-data
        $isMultipart = isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'multipart/form-data') !== false;
        if ($isMultipart) {
            $phone = $_POST['phone'] ?? null;
            $experience = $_POST['experience'] ?? null;
            $credentials = $_POST['credentials'] ?? null;
            $bio = $_POST['bio'] ?? null;
            $certification = null;
            if (isset($_FILES['certification']) && $_FILES['certification']['error'] === UPLOAD_ERR_OK) {
                $uploadsDir = __DIR__ . '/../uploads/';
                if (!is_dir($uploadsDir)) mkdir($uploadsDir, 0777, true);
                $filename = uniqid('cert_', true) . '_' . basename($_FILES['certification']['name']);
                $targetPath = $uploadsDir . $filename;
                if (move_uploaded_file($_FILES['certification']['tmp_name'], $targetPath)) {
                    $certification = '/uploads/' . $filename;
                }
            }
        } else {
            $phone = $input['phone'] ?? null;
            $experience = $input['experience'] ?? null;
            $credentials = $input['credentials'] ?? null;
            $bio = $input['bio'] ?? null;
            $certification = $input['certification'] ?? null;
        }
        return $manager->applyToBeBroker($user_id, $phone, $experience, $credentials, $bio, $certification);
    }
    // Get broker application status for current user
    if ($method === 'GET' && preg_match('#^/api/brokers/application-status$#', $path)) {
        if (!isset($headers['Authorization'])) return ['error' => 'Missing token', 'code' => 401];
        require_once __DIR__ . '/../utils/jwt.php';
        $token = str_replace('Bearer ', '', $headers['Authorization']);
        $payload = verify_jwt($token);
        if (!$payload) return ['error' => 'Invalid or expired token', 'code' => 401];
        $user_id = $payload['sub'];
        return $manager->getBrokerApplicationStatus($user_id);
    }

    // Admin: Approve/reject manager application
    if ($method === 'POST' && preg_match('#^/api/brokers/applications/(\d+)/approve$#', $path, $matches)) {
        if (!isset($headers['Authorization'])) return ['error' => 'Missing token', 'code' => 401];
        require_once __DIR__ . '/../utils/jwt.php';
        $token = str_replace('Bearer ', '', $headers['Authorization']);
        $payload = verify_jwt($token);
        if (!$payload) return ['error' => 'Invalid or expired token', 'code' => 401];
        if (!isset($payload['role']) || $payload['role'] !== 'admin') {
            return ['error' => 'Unauthorized', 'code' => 403];
        }

        $application_id = $matches[1];
        if (!isset($input['status']) || !in_array($input['status'], ['approved', 'rejected'])) {
            return ['error' => 'Invalid status', 'code' => 400];
        }

        return $manager->approveManagerApplication($application_id, $input['status'], $payload['sub']);
    }

    return ['error' => 'Not found', 'code' => 404];
} 