<?php
// Updated: 2025-08-19 11:37 - Force cache refresh
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE, PATCH');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../controllers/JointAccountController.php';
require_once __DIR__ . '/../utils/jwt.php';

function handleJointAccountRequest($method, $path, $data, $headers) {
    // Debug: Log the request
    error_log("JointAccountHandler: Method=$method, Path=$path");
    
    $controller = new JointAccountController();

    if (!isset($headers['Authorization'])) {
        return ['error' => 'Missing token', 'code' => 401];
    }
    $token = str_replace('Bearer ', '', $headers['Authorization']);
    $payload = verify_jwt($token);
    if (!$payload) {
        return ['error' => 'Invalid or expired token', 'code' => 401];
    }
    $user_id = $payload['sub'];
    
    // GET /api/joint-accounts - Get user's joint accounts
    if ($method === 'GET' && $path === '/api/joint-accounts') {
        return $controller->getUserJointAccounts($user_id);
    }
    
    // POST /api/joint-accounts/invite - Create invitation
    if ($method === 'POST' && $path === '/api/joint-accounts/invite') {
        return $controller->createInvitation($user_id, $data);
    }
    
    // POST /api/joint-accounts/validate-invitation - Validate invitation (public)
    if ($method === 'POST' && $path === '/api/joint-accounts/validate-invitation') {
        $token = $data['token'] ?? '';
        return $controller->validateInvitation($token);
    }
    
    // POST /api/joint-accounts/accept - Accept invitation
    if ($method === 'POST' && $path === '/api/joint-accounts/accept') {
        $token = $data['token'] ?? '';
        
        // If user is not authenticated, this might be called from signup flow
        if (!$user_id && isset($data['user_id'])) {
            $user_id = $data['user_id'];
        }
        
        if (!$user_id) {
            return ['error' => 'Authentication required'];
        }
        
        return $controller->acceptInvitation($user_id, $token);
    }
    
    // GET /api/joint-accounts/{id} - Get joint account details
    if ($method === 'GET' && preg_match('#^/api/joint-accounts/(\d+)$#', $path, $matches)) {
        $account_id = intval($matches[1]);
        return $controller->getJointAccountDetails($user_id, $account_id);
    }
    
    // POST /api/joint-accounts/transactions - Create transaction
    if ($method === 'POST' && $path === '/api/joint-accounts/transactions') {
        return $controller->createTransaction($user_id, $data);
    }
    
    // POST /api/joint-accounts/transactions/approve - Approve transaction
    if ($method === 'POST' && $path === '/api/joint-accounts/transactions/approve') {
        $transaction_id = $data['transaction_id'] ?? 0;
        return $controller->approveTransaction($user_id, $transaction_id);
    }
    
    // GET /api/joint-accounts/{id}/pending-transactions - Get pending transactions
    if ($method === 'GET' && preg_match('#^/api/joint-accounts/(\d+)/pending-transactions$#', $path, $matches)) {
        $account_id = intval($matches[1]);
        return $controller->getPendingTransactions($user_id, $account_id);
    }
    
    // DELETE /api/joint-accounts/{id}/members/{user_id} - Remove member
    if ($method === 'DELETE' && preg_match('#^/api/joint-accounts/(\d+)/members/(\d+)$#', $path, $matches)) {
        $account_id = intval($matches[1]);
        $member_user_id = intval($matches[2]);
        return $controller->removeMember($user_id, $account_id, $member_user_id);
    }

    // POST /api/joint-accounts/{id}/trade - Execute trade
    if ($method === 'POST' && preg_match('#^/api/joint-accounts/(\d+)/trade$#', $path, $matches)) {
        $account_id = intval($matches[1]);
        $data = array_merge($data, ['joint_account_id' => $account_id]);
        return $controller->executeTrade($user_id, $data);
    }

    // GET /api/joint-accounts/{id}/positions - Get positions
    if ($method === 'GET' && preg_match('#^/api/joint-accounts/(\d+)/positions$#', $path, $matches)) {
        $account_id = intval($matches[1]);
        return $controller->getPositions($user_id, $account_id);
    }

    // GET /api/joint-accounts/{id}/orders - Get orders
    if ($method === 'GET' && preg_match('#^/api/joint-accounts/(\d+)/orders$#', $path, $matches)) {
        $account_id = intval($matches[1]);
        return $controller->getOrders($user_id, $account_id);
    }

    // GET /api/joint-accounts/{id}/performance - Get performance
    if ($method === 'GET' && preg_match('#^/api/joint-accounts/(\d+)/performance$#', $path, $matches)) {
        $account_id = intval($matches[1]);
        return $controller->getPerformance($user_id, $account_id);
    }
    
    return ['error' => 'Method not allowed', 'code' => 405];
}
