<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE, PATCH');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../controllers/InvestmentController.php';
require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../utils/jwt.php';

function handleInvestmentRequest($method, $path, $input, $headers) {
    $controller = new InvestmentController();

    if (!isset($headers['Authorization'])) return ['error' => 'Missing token', 'code' => 401];
    $token = str_replace('Bearer ', '', $headers['Authorization']);
    $payload = verify_jwt($token);
    if (!$payload) return ['error' => 'Invalid or expired token', 'code' => 401];
    $user_id = $payload['sub'];

    if ($method === 'POST' && $path === '/api/investments') {
        $controller->buy($user_id);
        exit;
    }

    if ($method === 'GET' && $path === '/api/investments/portfolio') {
        $type = $_GET['type'] ?? null;
        $controller->getPortfolio($user_id, $type);
        exit;
    }

    if ($method === 'POST' && preg_match('#^/api/investments/status$#', $path)) {
        if (!isset($input['investment_id'], $input['status'])) {
            return ['error' => 'investment_id and status are required', 'code' => 400];
        }
        
        // Extract optional trading parameters
        $trading_type = isset($input['trading_type']) ? $input['trading_type'] : null;
        $trading_pairs = isset($input['trading_pairs']) ? $input['trading_pairs'] : null;
        $risk_level = isset($input['risk_level']) ? $input['risk_level'] : null;
        
        return $controller->updateInvestmentStatus(
            $user_id, 
            $input['investment_id'], 
            $input['status'],
            $trading_type,
            $trading_pairs,
            $risk_level
        );
    }

    if ($method === 'POST' && preg_match('#^/api/investments/pnl$#', $path)) {
        if (!isset($input['investment_id'], $input['pnl_amount'])) {
            return ['error' => 'investment_id and pnl_amount are required', 'code' => 400];
        }

        // Optional role enforcement: ensure caller is a manager
        if (!isset($payload['role']) || strtolower($payload['role']) !== 'manager') {
            // Not strictly necessary if you want admins to post as well; comment out to relax
            // return ['error' => 'Only managers can add PnL', 'code' => 403];
        }

        // Resolve manager id from token's user id (active manager section)
        $pdo = getPDO();
        $stmt = $pdo->prepare('SELECT id FROM managers WHERE user_id = ? LIMIT 1');
        $stmt->execute([$user_id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $manager_id = $row ? intval($row['id']) : null;

        // Pass manager identifier to controller; it accepts either manager.user_id or manager.id (fallback in controller)
        $manager_identifier = $manager_id ?: $user_id;
        return $controller->addPnL($manager_identifier, $input['investment_id'], $input['pnl_amount']);
    }

    if ($method === 'GET' && preg_match('#^/api/investments/(\d+)/pnl$#', $path, $matches)) {
        return $controller->getPnlHistory($matches[1]);
    }

    if ($method === 'GET' && $path === '/api/investments/user') {
        return $controller->getUserInvestments($user_id);
    }
    
    // GET /api/investments/{id} - Get specific investment details
    if ($method === 'GET' && preg_match('#^/api/investments/(\d+)$#', $path, $matches)) {
        $investment_id = $matches[1];
        return $controller->getInvestmentDetails($investment_id);
    }
    
    // POST /api/investments/update - Update investment details
    if ($method === 'POST' && $path === '/api/investments/update') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($input['investment_id'])) {
            return ['error' => 'Missing investment_id', 'code' => 400];
        }
        
        $investment_id = $input['investment_id'];
        unset($input['investment_id']); // Remove from data array
        
        return $controller->updateInvestment($investment_id, $input);
    }

    return ['error' => 'Not found', 'code' => 404];
} 