<?php
require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../utils/jwt.php';

function handleDemoFundRequest($method, $path, $input, $headers) {
    $pdo = getPDO();
    
    // Verify JWT token
    $user = verify_jwt_from_headers($headers);
    if (!$user) {
        return ['error' => 'Unauthorized', 'code' => 401];
    }

    switch ($method) {
        case 'GET':
            if (preg_match('#^/api/demo-fund-applications$#', $path)) {
                return getUserDemoFundApplications($pdo, $user['sub']);
            }
            break;
            
        case 'POST':
            if (preg_match('#^/api/demo-fund-applications$#', $path)) {
                return createDemoFundApplication($pdo, $user['sub'], $input);
            }
            break;
    }
    
    return ['error' => 'Endpoint not found', 'code' => 404];
}

function getUserDemoFundApplications($pdo, $user_id) {
    try {
        $stmt = $pdo->prepare('
            SELECT id, requested_amount, purpose, trading_experience, risk_tolerance, 
                   additional_notes, status, approved_amount, admin_notes, created_at
            FROM demo_fund_applications 
            WHERE user_id = ? 
            ORDER BY created_at DESC
        ');
        $stmt->execute([$user_id]);
        $applications = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return ['success' => true, 'applications' => $applications];
    } catch (Exception $e) {
        return ['error' => 'Failed to fetch applications: ' . $e->getMessage(), 'code' => 500];
    }
}

function createDemoFundApplication($pdo, $user_id, $input) {
    try {
        // Check if user already has a pending application
        $stmt = $pdo->prepare('SELECT id FROM demo_fund_applications WHERE user_id = ? AND status = "pending"');
        $stmt->execute([$user_id]);
        if ($stmt->fetch()) {
            return ['error' => 'You already have a pending demo fund application', 'code' => 400];
        }
        
        // Validate required fields
        if (!isset($input['requested_amount']) || !isset($input['purpose'])) {
            return ['error' => 'Requested amount and purpose are required', 'code' => 400];
        }
        
        $amount = floatval($input['requested_amount']);
        if ($amount < 1000 || $amount > 100000) {
            return ['error' => 'Requested amount must be between $1,000 and $100,000', 'code' => 400];
        }
        
        // Create application
        $stmt = $pdo->prepare('
            INSERT INTO demo_fund_applications 
            (user_id, requested_amount, purpose, trading_experience, risk_tolerance, additional_notes)
            VALUES (?, ?, ?, ?, ?, ?)
        ');
        $stmt->execute([
            $user_id,
            $amount,
            $input['purpose'],
            $input['trading_experience'] ?? null,
            $input['risk_tolerance'] ?? null,
            $input['additional_notes'] ?? null
        ]);
        
        $applicationId = $pdo->lastInsertId();
        
        return [
            'success' => true,
            'message' => 'Demo fund application submitted successfully',
            'application_id' => $applicationId
        ];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to create application: ' . $e->getMessage(), 'code' => 500];
    }
}

// Admin functions for managing demo fund applications
function handleAdminDemoFundRequest($method, $path, $input, $headers) {
    $pdo = getPDO();
    
    // Verify admin permissions
    $user = verify_jwt_from_headers($headers);
    if (!$user || $user['role'] !== 'admin') {
        return ['error' => 'Unauthorized - Admin access required', 'code' => 401];
    }

    switch ($method) {
        case 'GET':
            if (preg_match('#^/api/admin/demo-fund-applications$#', $path)) {
                return getAdminDemoFundApplications($pdo);
            }
            break;
            
        case 'PUT':
            if (preg_match('#^/api/admin/demo-fund-applications/(\d+)/approve$#', $path, $matches)) {
                return approveDemoFundApplication($pdo, $matches[1], $input, $user['sub']);
            } elseif (preg_match('#^/api/admin/demo-fund-applications/(\d+)/reject$#', $path, $matches)) {
                return rejectDemoFundApplication($pdo, $matches[1], $input, $user['sub']);
            }
            break;
    }
    
    return ['error' => 'Endpoint not found', 'code' => 404];
}

function getAdminDemoFundApplications($pdo) {
    try {
        $stmt = $pdo->prepare('
            SELECT dfa.*, u.full_name as user_name, u.email as user_email
            FROM demo_fund_applications dfa
            JOIN users u ON dfa.user_id = u.id
            ORDER BY dfa.created_at DESC
        ');
        $stmt->execute();
        $applications = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return ['success' => true, 'applications' => $applications];
    } catch (Exception $e) {
        return ['error' => 'Failed to fetch applications: ' . $e->getMessage(), 'code' => 500];
    }
}

function approveDemoFundApplication($pdo, $applicationId, $input, $adminId) {
    try {
        // Get the application
        $stmt = $pdo->prepare('SELECT * FROM demo_fund_applications WHERE id = ?');
        $stmt->execute([$applicationId]);
        $application = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$application) {
            return ['error' => 'Application not found', 'code' => 404];
        }
        
        $approvedAmount = $input['approved_amount'] ?? $application['requested_amount'];
        $adminNotes = $input['admin_notes'] ?? 'Demo fund application approved';
        
        // Update application status
        $stmt = $pdo->prepare('
            UPDATE demo_fund_applications 
            SET status = "approved", approved_amount = ?, admin_notes = ?, reviewed_by = ?, reviewed_at = NOW()
            WHERE id = ?
        ');
        $stmt->execute([$approvedAmount, $adminNotes, $adminId, $applicationId]);
        
        // Update user's demo balance
        $stmt = $pdo->prepare('UPDATE users SET demo_balance = demo_balance + ? WHERE id = ?');
        $stmt->execute([$approvedAmount, $application['user_id']]);
        
        return [
            'success' => true,
            'message' => 'Demo fund application approved and user balance updated',
            'approved_amount' => $approvedAmount
        ];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to approve application: ' . $e->getMessage(), 'code' => 500];
    }
}

function rejectDemoFundApplication($pdo, $applicationId, $input, $adminId) {
    try {
        $adminNotes = $input['admin_notes'] ?? 'Demo fund application rejected';
        
        $stmt = $pdo->prepare('
            UPDATE demo_fund_applications 
            SET status = "rejected", admin_notes = ?, reviewed_by = ?, reviewed_at = NOW()
            WHERE id = ?
        ');
        $stmt->execute([$adminNotes, $adminId, $applicationId]);
        
        return ['success' => true, 'message' => 'Demo fund application rejected'];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to reject application: ' . $e->getMessage(), 'code' => 500];
    }
}
?>
