<?php
require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../utils/jwt.php';

function handleAdminRequest($method, $path, $input, $headers) {
    // Verify admin permissions
    $user = verify_jwt_from_headers($headers);
    if (!$user || $user['role'] !== 'admin') {
        return ['error' => 'Unauthorized - Admin access required', 'code' => 401];
    }

    $pdo = getPDO();

    switch ($method) {
        case 'GET':
            if (preg_match('#^/api/admin/dashboard$#', $path)) {
                return getAdminDashboard($pdo);
            } elseif (preg_match('#^/api/admin/users$#', $path)) {
                return getUsers($pdo, $_GET);
            } elseif (preg_match('#^/api/admin/users/(\d+)$#', $path, $matches)) {
                return getUserDetails($pdo, $matches[1]);
            } elseif (preg_match('#^/api/admin/analytics$#', $path)) {
                return getAdminAnalytics($pdo);
            } elseif (preg_match('#^/api/admin/alerts$#', $path)) {
                return getAlerts($pdo);
            } elseif (preg_match('#^/api/admin/recent-activity$#', $path)) {
                return getRecentActivity($pdo);
            } elseif (preg_match('#^/api/admin/stock-suggestions$#', $path)) {
                return getStockSuggestions($pdo);
            } elseif (preg_match('#^/api/admin/stock-suggestions/(\d+)$#', $path, $matches)) {
                return getStockSuggestion($pdo, $matches[1]);
            } elseif (preg_match('#^/api/admin/payment-requests$#', $path)) {
                $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
                $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
                return getPaymentRequests($pdo, $page, $limit);
            } elseif (preg_match('#^/api/admin/payment-details$#', $path)) {
                return getPaymentDetails($pdo);
            } elseif (preg_match('#^/api/admin/payment-requests/(\d+)$#', $path, $matches)) {
                return getPaymentRequest($pdo, $matches[1]);
            } elseif (preg_match('#^/api/admin/payment-methods$#', $path)) {
                return getAdminPaymentMethods($pdo);
            }
            break;

        case 'POST':
            if (preg_match('#^/api/admin/users/(\d+)/suspend$#', $path, $matches)) {
                return suspendUser($pdo, $matches[1], $input);
            } elseif (preg_match('#^/api/admin/users/(\d+)/activate$#', $path, $matches)) {
                return activateUser($pdo, $matches[1]);
            } elseif (preg_match('#^/api/admin/users/(\d+)/delete$#', $path, $matches)) {
                return deleteUser($pdo, $matches[1]);
            } elseif (preg_match('#^/api/admin/stock-suggestions$#', $path)) {
                return createStockSuggestion($pdo, $input);
            } elseif (preg_match('#^/api/admin/payment-details$#', $path)) {
                return createPaymentDetails($pdo, $input);
            } elseif (preg_match('#^/api/admin/approve-deposit/(\d+)$#', $path, $matches)) {
                return approveDeposit($pdo, $matches[1], $input);
            } elseif (preg_match('#^/api/admin/payment-methods$#', $path)) {
                return createAdminPaymentMethod($pdo, $input);
            } elseif (preg_match('#^/api/admin/send-email$#', $path)) {
                return sendAdminEmail($pdo, $input);
            }
            break;

        case 'PUT':
            if (preg_match('#^/api/admin/stock-suggestions/(\d+)$#', $path, $matches)) {
                return updateStockSuggestion($pdo, $matches[1], $input);
            } elseif (preg_match('#^/api/admin/payment-requests/(\d+)$#', $path, $matches)) {
                return updatePaymentRequestStatus($pdo, $matches[1], $input);
            } elseif (preg_match('#^/api/admin/payment-methods/(\d+)$#', $path, $matches)) {
                return updateAdminPaymentMethod($pdo, $matches[1], $input);
            }
            break;

        case 'DELETE':
            if (preg_match('#^/api/admin/stock-suggestions/(\d+)$#', $path, $matches)) {
                return deleteStockSuggestion($pdo, $matches[1]);
            } elseif (preg_match('#^/api/admin/payment-methods/(\d+)$#', $path, $matches)) {
                return deleteAdminPaymentMethod($pdo, $matches[1]);
            }
            break;
    }

    return ['error' => 'Not found', 'code' => 404];
}

function getAdminDashboard($pdo) {
    try {
        $metrics = [];
        
        // Total users
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM users");
        $metrics['total_users'] = $stmt->fetch()['total'];
        
        // Active users (last 30 days)
        $stmt = $pdo->query("SELECT COUNT(DISTINCT user_id) as active FROM user_activity WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
        $metrics['active_users'] = $stmt->fetch()['active'];
        
        // Total managers
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM users WHERE role = 'manager'");
        $metrics['total_managers'] = $stmt->fetch()['total'];
        
        // Total transactions
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM transactions");
        $metrics['total_transactions'] = $stmt->fetch()['total'];
        
        // Total volume
        $stmt = $pdo->query("SELECT SUM(amount) as total FROM transactions WHERE status = 'completed'");
        $metrics['total_volume'] = $stmt->fetch()['total'] ?? 0;
        
        // Calculate percentage changes
        $metrics['user_growth_percentage'] = calculatePercentageChange($pdo, 'users', 'created_at');
        $metrics['active_users_percentage'] = calculateActiveUsersPercentageChange($pdo);
        $metrics['managers_percentage'] = calculatePercentageChange($pdo, 'users', 'created_at', "role = 'manager'");
        $metrics['volume_percentage'] = calculateVolumePercentageChange($pdo);
        
        return [
            'success' => true,
            'data' => [
                'metrics' => $metrics
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function calculatePercentageChange($pdo, $table, $dateColumn, $whereClause = '') {
    try {
        $where = $whereClause ? "WHERE $whereClause" : '';
        
        // Current month count
        $stmt = $pdo->query("SELECT COUNT(*) as current FROM $table $where AND $dateColumn >= DATE_FORMAT(NOW(), '%Y-%m-01')");
        $current = $stmt->fetch()['current'];
        
        // Last month count
        $stmt = $pdo->query("SELECT COUNT(*) as previous FROM $table $where AND $dateColumn >= DATE_FORMAT(DATE_SUB(NOW(), INTERVAL 1 MONTH), '%Y-%m-01') AND $dateColumn < DATE_FORMAT(NOW(), '%Y-%m-01')");
        $previous = $stmt->fetch()['previous'];
        
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        
        return round((($current - $previous) / $previous) * 100, 1);
    } catch (Exception $e) {
        return 0;
    }
}

function calculateActiveUsersPercentageChange($pdo) {
    try {
        // Current month active users
        $stmt = $pdo->query("SELECT COUNT(DISTINCT user_id) as current FROM user_activity WHERE created_at >= DATE_FORMAT(NOW(), '%Y-%m-01')");
        $current = $stmt->fetch()['current'];
        
        // Last month active users
        $stmt = $pdo->query("SELECT COUNT(DISTINCT user_id) as previous FROM user_activity WHERE created_at >= DATE_FORMAT(DATE_SUB(NOW(), INTERVAL 1 MONTH), '%Y-%m-01') AND created_at < DATE_FORMAT(NOW(), '%Y-%m-01')");
        $previous = $stmt->fetch()['previous'];
        
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        
        return round((($current - $previous) / $previous) * 100, 1);
    } catch (Exception $e) {
        return 0;
    }
}

function calculateVolumePercentageChange($pdo) {
    try {
        // Current month volume
        $stmt = $pdo->query("SELECT SUM(amount) as current FROM transactions WHERE status = 'completed' AND created_at >= DATE_FORMAT(NOW(), '%Y-%m-01')");
        $current = $stmt->fetch()['current'] ?? 0;
        
        // Last month volume
        $stmt = $pdo->query("SELECT SUM(amount) as previous FROM transactions WHERE status = 'completed' AND created_at >= DATE_FORMAT(DATE_SUB(NOW(), INTERVAL 1 MONTH), '%Y-%m-01') AND created_at < DATE_FORMAT(NOW(), '%Y-%m-01')");
        $previous = $stmt->fetch()['previous'] ?? 0;
        
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        
        return round((($current - $previous) / $previous) * 100, 1);
    } catch (Exception $e) {
        return 0;
    }
}

function getUsers($pdo, $params) {
    try {
        $page = isset($params['page']) ? (int)$params['page'] : 1;
        $limit = isset($params['limit']) ? (int)$params['limit'] : 20;
        $search = isset($params['search']) ? $params['search'] : '';
        $role = isset($params['role']) ? $params['role'] : '';
        
        $offset = ($page - 1) * $limit;
        
        $where_conditions = [];
        $query_params = [];
        
        if ($search) {
            $where_conditions[] = "(full_name LIKE ? OR email LIKE ?)";
            $search_param = "%$search%";
            $query_params[] = $search_param;
            $query_params[] = $search_param;
        }
        
        if ($role) {
            $where_conditions[] = "role = ?";
            $query_params[] = $role;
        }
        
        $where_clause = $where_conditions ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
        
        // Get total count
        $count_sql = "SELECT COUNT(*) as total FROM users $where_clause";
        $stmt = $pdo->prepare($count_sql);
        $stmt->execute($query_params);
        $total = $stmt->fetch()['total'];
        
        // Get users with explicit LIMIT and OFFSET binding
        $sql = "
            SELECT id, full_name as name, email, email as username, role, COALESCE(status, 'active') as status, created_at, created_at as last_login
            FROM users 
            $where_clause
            ORDER BY created_at DESC 
            LIMIT $limit OFFSET $offset
        ";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($query_params);
        $users = $stmt->fetchAll();
        
        return [
            'success' => true,
            'data' => [
                'users' => $users,
                'pagination' => [
                    'page' => $page,
                    'limit' => $limit,
                    'total' => $total,
                    'pages' => ceil($total / $limit)
                ]
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getUserDetails($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("
            SELECT id, name, email, username, role, status, created_at, last_login
            FROM users 
            WHERE id = ?
        ");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return ['error' => 'User not found', 'code' => 404];
        }
        
        return [
            'success' => true,
            'data' => $user
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function suspendUser($pdo, $userId, $input) {
    try {
        $stmt = $pdo->prepare("UPDATE users SET status = 'suspended' WHERE id = ?");
        $stmt->execute([$userId]);
        
        return ['success' => true, 'message' => 'User suspended successfully'];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function activateUser($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("UPDATE users SET status = 'active' WHERE id = ?");
        $stmt->execute([$userId]);
        
        return ['success' => true, 'message' => 'User activated successfully'];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function deleteUser($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("UPDATE users SET status = 'deleted', deleted_at = NOW() WHERE id = ?");
        $stmt->execute([$userId]);
        
        return ['success' => true, 'message' => 'User deleted successfully'];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getAdminAnalytics($pdo) {
    try {
        $analytics = [];
        
        // User growth over time
        $stmt = $pdo->query("
            SELECT DATE(created_at) as date, COUNT(*) as new_users
            FROM users 
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY DATE(created_at)
            ORDER BY date
        ");
        $analytics['user_growth'] = $stmt->fetchAll();
        
        // Users by role
        $stmt = $pdo->query("
            SELECT role, COUNT(*) as count
            FROM users 
            WHERE status = 'active'
            GROUP BY role
        ");
        $analytics['users_by_role'] = $stmt->fetchAll();
        
        return [
            'success' => true,
            'data' => $analytics
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getAlerts($pdo) {
    try {
        // For now, return empty alerts - we'll implement this later
        return [
            'success' => true,
            'data' => []
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getRecentActivity($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT 
                'registration' as type,
                'New user registration' as description,
                u.email as user_email,
                u.name as user_name,
                NULL as amount,
                u.created_at
            FROM users u
            WHERE u.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            
            UNION ALL
            
            SELECT 
                'transaction' as type,
                CONCAT('Transaction: $', FORMAT(t.amount, 0)) as description,
                u.email as user_email,
                u.name as user_name,
                t.amount,
                t.created_at
            FROM transactions t
            JOIN users u ON t.user_id = u.id
            WHERE t.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            AND t.amount > 1000
            
            UNION ALL
            
            SELECT 
                'manager_application' as type,
                'Manager application submitted' as description,
                u.email as user_email,
                u.name as user_name,
                NULL as amount,
                u.created_at
            FROM users u
            WHERE u.role = 'manager' 
            AND u.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            
            ORDER BY created_at DESC
            LIMIT 10
        ");
        
        $activities = $stmt->fetchAll();
        
        return [
            'success' => true,
            'data' => [
                'activities' => $activities
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getStockSuggestions($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT 
                s.*,
                COUNT(ua.suggestion_id) as adoption_count
            FROM stock_suggestions s
            LEFT JOIN user_adoptions ua ON s.id = ua.suggestion_id
            GROUP BY s.id
            ORDER BY s.created_at DESC
        ");
        
        $suggestions = $stmt->fetchAll();
        
        return [
            'success' => true,
            'data' => [
                'suggestions' => $suggestions
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getStockSuggestion($pdo, $id) {
    try {
        $stmt = $pdo->prepare("
            SELECT 
                s.*,
                COUNT(ua.suggestion_id) as adoption_count
            FROM stock_suggestions s
            LEFT JOIN user_adoptions ua ON s.id = ua.suggestion_id
            WHERE s.id = ?
            GROUP BY s.id
        ");
        $stmt->execute([$id]);
        $suggestion = $stmt->fetch();
        
        if (!$suggestion) {
            return ['error' => 'Stock suggestion not found', 'code' => 404];
        }
        
        return [
            'success' => true,
            'data' => $suggestion
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function createStockSuggestion($pdo, $input) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO stock_suggestions (symbol, name, reason, target_price, current_price, risk_level, asset_type, status, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'active', NOW())
        ");
        
        $stmt->execute([
            $input['symbol'],
            $input['name'],
            $input['reason'],
            $input['target_price'],
            $input['current_price'],
            $input['risk_level'],
            $input['asset_type'] ?? 'stock' // Default to 'stock' if not provided
        ]);
        
        return [
            'success' => true,
            'message' => 'Stock suggestion created successfully',
            'data' => ['id' => $pdo->lastInsertId()]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function updateStockSuggestion($pdo, $id, $input) {
    try {
        $stmt = $pdo->prepare("
            UPDATE stock_suggestions 
            SET symbol = ?, name = ?, reason = ?, target_price = ?, current_price = ?, risk_level = ?, asset_type = ?
            WHERE id = ?
        ");
        
        $stmt->execute([
            $input['symbol'],
            $input['name'],
            $input['reason'],
            $input['target_price'],
            $input['current_price'],
            $input['risk_level'],
            $input['asset_type'] ?? 'stock', // Default to 'stock' if not provided
            $id
        ]);
        
        return [
            'success' => true,
            'message' => 'Stock suggestion updated successfully'
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function deleteStockSuggestion($pdo, $id) {
    try {
        $stmt = $pdo->prepare("DELETE FROM stock_suggestions WHERE id = ?");
        $stmt->execute([$id]);
        
        return [
            'success' => true,
            'message' => 'Stock suggestion deleted successfully'
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getPaymentRequests($pdo, $page = 1, $limit = 20) {
    try {
        $offset = ($page - 1) * $limit;
        
        // Get total count
        $countStmt = $pdo->query("
            SELECT COUNT(*) as total
            FROM payment_requests pr
            JOIN users u ON pr.user_id = u.id
        ");
        $totalCount = $countStmt->fetch()['total'];
        
        // Get paginated results
        $stmt = $pdo->prepare("
            SELECT 
                pr.*,
                u.full_name as user_name,
                u.email as user_email,
                pm.name as payment_method_name,
                pm.type as payment_method_type
            FROM payment_requests pr
            JOIN users u ON pr.user_id = u.id
            LEFT JOIN payment_methods pm ON JSON_EXTRACT(pr.payment_details, '$.payment_method_id') = pm.id
            ORDER BY pr.created_at DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->bindValue(1, (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(2, (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        $requests = $stmt->fetchAll();
        
        // Decode payment_details JSON for each request
        foreach ($requests as &$request) {
            if ($request['payment_details']) {
                $details = json_decode($request['payment_details'], true);
                $request['payment_details'] = $details;
                // Extract additional fields for compatibility
                $request['currency'] = $details['currency'] ?? 'USD';
                $request['reference_number'] = $details['reference_number'] ?? '';
                $request['user_notes'] = $details['notes'] ?? '';
                $request['proof_file'] = $details['proof_file'] ?? null;
            }
        }
        
        return [
            'success' => true,
            'data' => [
                'requests' => $requests,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $limit,
                    'total' => $totalCount,
                    'total_pages' => ceil($totalCount / $limit)
                ]
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getPaymentRequest($pdo, $id) {
    try {
        $stmt = $pdo->prepare("
            SELECT 
                pr.*,
                u.name as user_name,
                u.email as user_email
            FROM payment_requests pr
            JOIN users u ON pr.user_id = u.id
            WHERE pr.id = ?
        ");
        $stmt->execute([$id]);
        $request = $stmt->fetch();
        
        if (!$request) {
            return ['error' => 'Payment request not found', 'code' => 404];
        }
        
        return [
            'success' => true,
            'data' => $request
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function updatePaymentRequest($pdo, $id, $input) {
    try {
        $stmt = $pdo->prepare("
            UPDATE payment_requests 
            SET status = ?, admin_notes = ?, payment_details = ?, updated_at = NOW()
            WHERE id = ?
        ");
        
        $stmt->execute([
            $input['status'],
            $input['admin_notes'] ?? null,
            $input['payment_details'] ?? null,
            $id
        ]);
        
        return [
            'success' => true,
            'message' => 'Payment request updated successfully'
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getPaymentDetails($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT * FROM payment_details
            ORDER BY created_at DESC
        ");
        
        $details = $stmt->fetchAll();
        
        return [
            'success' => true,
            'data' => [
                'details' => $details
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function createPaymentDetails($pdo, $input) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO payment_details (bank_name, account_number, routing_number, account_holder, instructions, is_active, created_at)
            VALUES (?, ?, ?, ?, ?, 1, NOW())
        ");
        
        $stmt->execute([
            $input['bank_name'],
            $input['account_number'],
            $input['routing_number'],
            $input['account_holder'],
            $input['instructions']
        ]);
        
        return [
            'success' => true,
            'message' => 'Payment details created successfully',
            'data' => ['id' => $pdo->lastInsertId()]
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function approveDeposit($pdo, $requestId, $input) {
    try {
        // Get the payment request
        $stmt = $pdo->prepare('SELECT * FROM payment_requests WHERE id = ?');
        $stmt->execute([$requestId]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            return ['error' => 'Payment request not found', 'code' => 404];
        }
        
        $approvedAmount = $input['approved_amount'] ?? $request['amount'];
        $adminNotes = $input['admin_notes'] ?? 'Deposit approved';
        
        // Update payment request status to approved
        $stmt = $pdo->prepare('UPDATE payment_requests SET status = ?, admin_notes = ? WHERE id = ?');
        $stmt->execute(['approved', $adminNotes, $requestId]);
        
        // Update user balance (assuming there's a user_balances table or users table with balance column)
        // For now, we'll create a simple balance record
        $stmt = $pdo->prepare('
            INSERT INTO user_balances (user_id, balance, last_updated) 
            VALUES (?, ?, NOW()) 
            ON DUPLICATE KEY UPDATE 
            balance = balance + ?, last_updated = NOW()
        ');
        $stmt->execute([$request['user_id'], $approvedAmount, $approvedAmount]);
        
        return [
            'success' => true,
            'message' => 'Deposit approved and balance updated',
            'approved_amount' => $approvedAmount,
            'request_id' => $requestId
        ];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to approve deposit: ' . $e->getMessage(), 'code' => 500];
    }
}

function updatePaymentRequestStatus($pdo, $requestId, $input) {
    try {
        // Get the payment request
        $stmt = $pdo->prepare("SELECT * FROM payment_requests WHERE id = ?");
        $stmt->execute([$requestId]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            return ['error' => 'Payment request not found', 'code' => 404];
        }
        
        if ($request['status'] !== 'pending') {
            return ['error' => 'Payment request is not pending', 'code' => 400];
        }
        
        // Update the request status
        $stmt = $pdo->prepare("UPDATE payment_requests SET status = ?, admin_notes = ?, updated_at = NOW() WHERE id = ?");
        $stmt->execute([$input['status'], $input['admin_notes'] ?? '', $requestId]);
        
        // If approved, update user balance using unified balance management
        if ($input['status'] === 'approved') {
            // Use the PaymentController's unified balance update method
            require_once __DIR__ . '/../controllers/PaymentController.php';
            $paymentController = new PaymentController();
            $balanceResult = $paymentController->updateUserBalance($request['user_id'], $request['amount'], 'add');
            
            if (!$balanceResult['success']) {
                error_log("Failed to update balance for user {$request['user_id']}: " . ($balanceResult['error'] ?? 'Unknown error'));
            }
        }
        
        return [
            'success' => true,
            'message' => 'Payment request updated successfully'
        ];
    } catch (Exception $e) {
        return ['error' => 'Database error: ' . $e->getMessage(), 'code' => 500];
    }
}

function getAdminPaymentMethods($pdo) {
    try {
        $stmt = $pdo->query('
            SELECT id, name, type, description, instructions, account_details, min_amount, max_amount, processing_time, fees, is_active, created_at
            FROM payment_methods 
            ORDER BY created_at DESC
        ');
        $methods = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Decode JSON fields
        foreach ($methods as &$method) {
            if (isset($method['account_details'])) {
                $method['account_details'] = json_decode($method['account_details'], true);
            }
        }
        
        return [
            'success' => true,
            'data' => [
                'payment_methods' => $methods
            ]
        ];
    } catch (Exception $e) {
        return ['error' => 'Failed to get payment methods: ' . $e->getMessage(), 'code' => 500];
    }
}

function createAdminPaymentMethod($pdo, $input) {
    try {
        $stmt = $pdo->prepare('
            INSERT INTO payment_methods (name, type, description, instructions, account_details, min_amount, max_amount, processing_time, fees)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ');
        $stmt->execute([
            $input['name'],
            $input['type'],
            $input['description'],
            $input['instructions'],
            json_encode($input['account_details']),
            $input['min_amount'],
            $input['max_amount'],
            $input['processing_time'],
            $input['fees']
        ]);
        
        return ['success' => true, 'message' => 'Payment method created successfully'];
    } catch (Exception $e) {
        return ['error' => 'Failed to create payment method: ' . $e->getMessage(), 'code' => 500];
    }
}

function updateAdminPaymentMethod($pdo, $methodId, $input) {
    try {
        $stmt = $pdo->prepare('
            UPDATE payment_methods 
            SET name = ?, type = ?, description = ?, instructions = ?, account_details = ?, 
                min_amount = ?, max_amount = ?, processing_time = ?, fees = ?, is_active = ?
            WHERE id = ?
        ');
        $stmt->execute([
            $input['name'],
            $input['type'],
            $input['description'],
            $input['instructions'],
            json_encode($input['account_details']),
            $input['min_amount'],
            $input['max_amount'],
            $input['processing_time'],
            $input['fees'],
            $input['is_active'] ?? true,
            $methodId
        ]);
        
        return ['success' => true, 'message' => 'Payment method updated successfully'];
    } catch (Exception $e) {
        return ['error' => 'Failed to update payment method: ' . $e->getMessage(), 'code' => 500];
    }
}

function deleteAdminPaymentMethod($pdo, $methodId) {
    try {
        $stmt = $pdo->prepare('DELETE FROM payment_methods WHERE id = ?');
        $stmt->execute([$methodId]);
        
        return ['success' => true, 'message' => 'Payment method deleted successfully'];
    } catch (Exception $e) {
        return ['error' => 'Failed to delete payment method: ' . $e->getMessage(), 'code' => 500];
    }
}

function sendAdminEmail($pdo, $input) {
    try {
        // In a real implementation, you'd integrate with an email service like SendGrid, Mailgun, etc.
        // For now, we'll log the email and return success
        
        $userId = $input['user_id'];
        $subject = $input['subject'];
        $message = $input['message'];
        $template = $input['template'] ?? 'custom';
        
        // Get user email
        $stmt = $pdo->prepare('SELECT email, full_name FROM users WHERE id = ?');
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return ['error' => 'User not found', 'code' => 404];
        }
        
        // Log the email (in production, replace with actual email sending)
        error_log("Email would be sent to: {$user['email']} with subject: {$subject}");
        
        // Store email record in database for tracking
        try {
            $stmt = $pdo->prepare('
                INSERT INTO admin_emails (user_id, subject, message, template, sent_at)
                VALUES (?, ?, ?, ?, NOW())
            ');
            $stmt->execute([$userId, $subject, $message, $template]);
        } catch (Exception $e) {
            // Table might not exist, ignore for now
        }
        
        return [
            'success' => true,
            'message' => 'Email sent successfully',
            'recipient' => $user['email']
        ];
        
    } catch (Exception $e) {
        return ['error' => 'Failed to send email: ' . $e->getMessage(), 'code' => 500];
    }
}
?>
