<?php
require_once __DIR__ . '/../utils/db.php';

class TransactionController {
    private $pdo;
    public function __construct() {
        $this->pdo = getPDO();
    }

    public function getUserTransactions($user_id) {
        $stmt = $this->pdo->prepare('SELECT * FROM transactions WHERE user_id = ?');
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function createTransaction($user_id, $data) {
        // Start database transaction for atomicity
        $this->pdo->beginTransaction();
        
        try {
            // For Buy transactions, deduct from funded balance
            if (strtolower($data['type']) === 'buy') {
                // Get user's current funded balance
                $stmt = $this->pdo->prepare("SELECT funded_balance FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$user) {
                    $this->pdo->rollBack();
                    return ['error' => 'User not found', 'code' => 404];
                }
                
                $balance = floatval($user['funded_balance']);
                $amount = floatval($data['amount']);
                
                if ($amount > $balance) {
                    $this->pdo->rollBack();
                    return ['error' => 'Insufficient funded balance', 'code' => 400];
                }
                
                // Deduct the amount from funded balance
                $newBalance = $balance - $amount;
                $updateStmt = $this->pdo->prepare("UPDATE users SET funded_balance = ? WHERE id = ?");
                $updateStmt->execute([$newBalance, $user_id]);
            }
            
            // Create transaction record
            $stmt = $this->pdo->prepare('INSERT INTO transactions (user_id, portfolio_id, asset_id, type, amount, quantity, price, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');
            $stmt->execute([
                $user_id,
                $data['portfolio_id'],
                $data['asset_id'],
                $data['type'],
                $data['amount'],
                $data['quantity'],
                $data['price'],
                $data['status'] ?? 'completed'
            ]);
            
            $transaction_id = $this->pdo->lastInsertId();
            
            // Commit the transaction
            $this->pdo->commit();
            
            $result = ['success' => true, 'transaction_id' => $transaction_id];
            
            // Add new balance to response if it was a Buy transaction
            if (strtolower($data['type']) === 'buy') {
                $result['new_funded_balance'] = $newBalance;
            }
            
            return $result;
            
        } catch (Exception $e) {
            // Rollback on any error
            $this->pdo->rollBack();
            return ['error' => 'Failed to create transaction: ' . $e->getMessage(), 'code' => 500];
        }
    }

    // Add update/delete methods as needed
} 