<?php

require_once __DIR__ . '/../utils/db.php';

class PortfolioController {
    private $pdo;

    public function __construct() {
        $this->pdo = getPDO();
    }

    public function getSummary($userId) {
        try {
            // 1. Get Wallet Balance
            $stmt = $this->pdo->prepare("SELECT funded_balance FROM users WHERE id = ?");
            $stmt->execute([$userId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            $cashBalance = $user ? floatval($user['funded_balance']) : 0;

            // 2. Get Standard Investments (Stocks, Crypto bought via InvestmentController)
            // Check if investments table exists first to avoid 500 error if migration not run
            try {
                $stmt = $this->pdo->prepare("SELECT SUM(current_value) as total_std FROM investments WHERE user_id = ? AND status = 'active'");
                $stmt->execute([$userId]);
                $stdInv = $stmt->fetch(PDO::FETCH_ASSOC);
                $stdTotal = $stdInv ? floatval($stdInv['total_std']) : 0;
            } catch (Exception $e) {
                // Return 0 if table missing, or handle graceful degradation
                $stdTotal = 0; 
                // Log error or just proceed? ideally we proceed with partial data but warn
            }

            // 3. Get Special Plans (Auto-Invest)
            $stmt = $this->pdo->prepare("SELECT SUM(current_value) as total_special, SUM(amount) as invested_special FROM user_special_investments WHERE user_id = ? AND status IN ('active', 'completed')");
            $stmt->execute([$userId]);
            $specialInv = $stmt->fetch(PDO::FETCH_ASSOC);
            $specialTotal = $specialInv ? floatval($specialInv['total_special']) : 0;
            $specialInvested = $specialInv ? floatval($specialInv['invested_special']) : 0;

            // 4. Calculate Net Worth & PnL
            $totalInvested = 0;
            $stdCost = 0;
            
            try {
                $stmt = $this->pdo->prepare("SELECT SUM(purchase_price * quantity) as cost_basis FROM investments WHERE user_id = ? AND status = 'active'");
                $stmt->execute([$userId]);
                $costBasis = $stmt->fetch(PDO::FETCH_ASSOC);
                $stdCost = $costBasis ? floatval($costBasis['cost_basis']) : 0;
            } catch (Exception $e) { $stdCost = 0; }

            // Count active
            $stdCount = 0;
            try {
                $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM investments WHERE user_id = ? AND status = 'active'");
                $stmt->execute([$userId]);
                $stdCount = $stmt->fetchColumn();
            } catch (Exception $e) { $stdCount = 0; }
            
            $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM user_special_investments WHERE user_id = ? AND status = 'active'");
            $stmt->execute([$userId]);
            $specialCount = $stmt->fetchColumn();

            $activeCount = $stdCount + $specialCount;

            $totalPortfolioValue = $stdTotal + $specialTotal;
            $totalNetWorth = $cashBalance + $totalPortfolioValue;
            
            $totalCost = $stdCost + $specialInvested;
            $totalPnL = $totalPortfolioValue - $totalCost;
            $pnlPercent = ($totalCost > 0) ? ($totalPnL / $totalCost) * 100 : 0;

            echo json_encode([
                'success' => true,
                'net_worth' => $totalNetWorth,
                'cash_balance' => $cashBalance,
                'total_invested' => $totalCost, // Cost basis
                'portfolio_value' => $totalPortfolioValue, // Current Value
                'total_pnl' => $totalPnL,
                'pnl_percent' => $pnlPercent,
                'active_count' => $activeCount,
                'breakdown' => [
                    'cash' => $cashBalance,
                    'stocks_crypto' => $stdTotal,
                    'auto_invest' => $specialTotal
                ]
            ]);
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Portfolio calculation error: ' . $e->getMessage()]);
        }
    }
}
