<?php
require_once __DIR__ . '/../utils/db.php';

class PaymentController {
    private $pdo;
    public function __construct() { $this->pdo = getPDO(); }

    public function getUserPayments($user_id) {
        $stmt = $this->pdo->prepare('SELECT * FROM payment_requests WHERE user_id = ? ORDER BY created_at DESC');
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function createPayment($user_id, $data) {
        $stmt = $this->pdo->prepare('INSERT INTO payment_requests (user_id, type, amount, status, payment_details, admin_notes) VALUES (?, ?, ?, ?, ?, ?)');
        $stmt->execute([
            $user_id,
            $data['type'] ?? 'deposit',
            $data['amount'],
            $data['status'] ?? 'pending',
            json_encode([
                'method' => $data['method'] ?? 'wire_transfer',
                'currency' => $data['currency'] ?? 'USD',
                'reference' => $data['reference'] ?? uniqid('PAY'),
                'notes' => $data['notes'] ?? null
            ]),
            null // admin_notes
        ]);
        return ['success' => true, 'payment_id' => $this->pdo->lastInsertId()];
    }

    public function updatePaymentStatus($payment_id, $status) {
        $stmt = $this->pdo->prepare('UPDATE payment_requests SET status = ? WHERE id = ?');
        $stmt->execute([$status, $payment_id]);
        return ['success' => true];
    }

    public function getPaymentsByStatus($user_id, $status) {
        $stmt = $this->pdo->prepare('SELECT * FROM payment_requests WHERE user_id = ? AND status = ? ORDER BY created_at DESC');
        $stmt->execute([$user_id, $status]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function refundPayment($payment_id) {
        // Example: mark as refunded, integrate with payment provider if needed
        $stmt = $this->pdo->prepare('UPDATE payment_requests SET status = ? WHERE id = ?');
        $stmt->execute(['refunded', $payment_id]);
        // Here you would call the payment provider API
        return ['success' => true, 'refunded' => true];
    }

    public function processWithProvider($user_id, $amount, $method) {
        // Stub for payment provider integration (e.g., Stripe, PayPal)
        // You would call the provider API here and return the result
        return ['success' => true, 'provider' => $method, 'amount' => $amount];
    }

    /**
     * Unified balance management - ensures both balance systems are synchronized
     */
    public function getUserBalance($user_id) {
        try {
            // Get balance from user_balances table (primary source)
            $stmt = $this->pdo->prepare('SELECT balance FROM user_balances WHERE user_id = ?');
            $stmt->execute([$user_id]);
            $userBalanceRow = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Get balance from users table (secondary source)
            $stmt = $this->pdo->prepare('SELECT funded_balance FROM users WHERE id = ?');
            $stmt->execute([$user_id]);
            $usersBalanceRow = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $userBalance = $userBalanceRow ? floatval($userBalanceRow['balance']) : 0;
            $usersBalance = $usersBalanceRow ? floatval($usersBalanceRow['funded_balance']) : 0;
            
            // If balances don't match, sync them
            if (abs($userBalance - $usersBalance) > 0.01) { // Allow for small floating point differences
                $this->syncUserBalances($user_id, $userBalance, $usersBalance);
            }
            
            return ['balance' => $userBalance];
        } catch (Exception $e) {
            // Return a default balance instead of error to prevent redirects
            return ['balance' => 0];
        }
    }

    /**
     * Get both funded and demo balances for a user
     */
    public function getUserBalances($user_id) {
        try {
            // Get funded balance from user_balances table (primary source)
            $stmt = $this->pdo->prepare('SELECT balance FROM user_balances WHERE user_id = ?');
            $stmt->execute([$user_id]);
            $userBalanceRow = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Get demo balance from users table
            $stmt = $this->pdo->prepare('SELECT demo_balance FROM users WHERE id = ?');
            $stmt->execute([$user_id]);
            $demoBalanceRow = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $fundedBalance = $userBalanceRow ? floatval($userBalanceRow['balance']) : 0;
            $demoBalance = $demoBalanceRow ? floatval($demoBalanceRow['demo_balance']) : 0;
            
            return [
                'success' => true,
                'data' => [
                    'funded_balance' => $fundedBalance,
                    'demo_balance' => $demoBalance
                ]
            ];
        } catch (Exception $e) {
            return [
                'success' => true,
                'data' => [
                    'funded_balance' => 0,
                    'demo_balance' => 0
                ]
            ];
        }
    }
    
    /**
     * Sync balances between user_balances and users tables
     */
    public function syncUserBalances($user_id, $userBalance, $usersBalance) {
        try {
            // Use the higher balance as the source of truth (most recent)
            $correctBalance = max($userBalance, $usersBalance);
            
            // Update user_balances table
            $stmt = $this->pdo->prepare('
                INSERT INTO user_balances (user_id, balance, last_updated) 
                VALUES (?, ?, NOW())
                ON DUPLICATE KEY UPDATE 
                balance = VALUES(balance),
                last_updated = NOW()
            ');
            $stmt->execute([$user_id, $correctBalance]);
            
            // Update users table
            $stmt = $this->pdo->prepare('UPDATE users SET funded_balance = ? WHERE id = ?');
            $stmt->execute([$correctBalance, $user_id]);
            
            error_log("Balance synced for user $user_id: user_balances=$userBalance, users=$usersBalance, corrected=$correctBalance");
            
        } catch (Exception $e) {
            error_log("Failed to sync balances for user $user_id: " . $e->getMessage());
        }
    }
    
    /**
     * Update user balance in both tables atomically
     */
    public function updateUserBalance($user_id, $amount, $operation = 'add') {
        try {
            $this->pdo->beginTransaction();
            
            // Get current balance
            $stmt = $this->pdo->prepare('SELECT balance FROM user_balances WHERE user_id = ?');
            $stmt->execute([$user_id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $currentBalance = $row ? floatval($row['balance']) : 0;
            
            // Calculate new balance
            $newBalance = $operation === 'add' ? $currentBalance + $amount : $currentBalance - $amount;
            
            // Update user_balances table
            $stmt = $this->pdo->prepare('
                INSERT INTO user_balances (user_id, balance, last_updated) 
                VALUES (?, ?, NOW())
                ON DUPLICATE KEY UPDATE 
                balance = VALUES(balance),
                last_updated = NOW()
            ');
            $stmt->execute([$user_id, $newBalance]);
            
            // Update users table
            $stmt = $this->pdo->prepare('UPDATE users SET funded_balance = ? WHERE id = ?');
            $stmt->execute([$newBalance, $user_id]);
            
            $this->pdo->commit();
            
            return [
                'success' => true,
                'old_balance' => $currentBalance,
                'new_balance' => $newBalance,
                'amount' => $amount,
                'operation' => $operation
            ];
            
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("Failed to update balance for user $user_id: " . $e->getMessage());
            return ['error' => 'Failed to update balance: ' . $e->getMessage()];
        }
    }
    
    /**
     * Validate and fix balance discrepancies
     */
    public function validateUserBalances($user_id) {
        try {
            // Get balance from user_balances table
            $stmt = $this->pdo->prepare('SELECT balance FROM user_balances WHERE user_id = ?');
            $stmt->execute([$user_id]);
            $userBalanceRow = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Get balance from users table
            $stmt = $this->pdo->prepare('SELECT funded_balance FROM users WHERE id = ?');
            $stmt->execute([$user_id]);
            $usersBalanceRow = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Calculate actual balance from payment_requests
            $stmt = $this->pdo->prepare('
                SELECT 
                    COALESCE(SUM(CASE WHEN type = "deposit" AND status IN ("approved", "completed") THEN amount ELSE 0 END), 0) -
                    COALESCE(SUM(CASE WHEN type = "withdrawal" AND status IN ("approved", "completed") THEN amount ELSE 0 END), 0) as actual_balance 
                FROM payment_requests 
                WHERE user_id = ?
            ');
            $stmt->execute([$user_id]);
            $actualRow = $stmt->fetch(PDO::FETCH_ASSOC);
            $actualBalance = floatval($actualRow['actual_balance'] ?? 0);
            
            $userBalance = $userBalanceRow ? floatval($userBalanceRow['balance']) : 0;
            $usersBalance = $usersBalanceRow ? floatval($usersBalanceRow['funded_balance']) : 0;
            
            $discrepancies = [];
            if (abs($userBalance - $actualBalance) > 0.01) {
                $discrepancies[] = "user_balances table: $userBalance vs actual: $actualBalance";
            }
            if (abs($usersBalance - $actualBalance) > 0.01) {
                $discrepancies[] = "users.funded_balance: $usersBalance vs actual: $actualBalance";
            }
            
            if (!empty($discrepancies)) {
                // Fix discrepancies by updating both tables to actual balance
                $this->syncUserBalances($user_id, $actualBalance, $actualBalance);
                
                return [
                    'success' => true,
                    'discrepancies_found' => true,
                    'discrepancies' => $discrepancies,
                    'corrected_balance' => $actualBalance
                ];
            }
            
            return [
                'success' => true,
                'discrepancies_found' => false,
                'balances_match' => true,
                'balance' => $actualBalance
            ];
            
        } catch (Exception $e) {
            return ['error' => 'Failed to validate balances: ' . $e->getMessage()];
        }
    }
} 