<?php
require_once __DIR__ . '/../utils/db.php';

class MessageController {
    private $pdo;
    
    public function __construct() {
        $this->pdo = getPDO();
    }

    public function getMessages($user_id, $other_user_id) {
        try {
            $stmt = $this->pdo->prepare('
                SELECT
                    m.id,
                    m.sender_id,
                    m.receiver_id,
                    m.message,
                    m.created_at,
                    sender.full_name as sender_name,
                    receiver.full_name as receiver_name
                FROM messages m
                LEFT JOIN users sender ON m.sender_id = sender.id
                LEFT JOIN users receiver ON m.receiver_id = receiver.id
                WHERE (m.sender_id = ? AND m.receiver_id = ?) 
                   OR (m.sender_id = ? AND m.receiver_id = ?)
                ORDER BY m.created_at ASC
            ');
            $stmt->execute([$user_id, $other_user_id, $other_user_id, $user_id]);
            $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'data' => $messages
            ];
        } catch (Exception $e) {
            return ['error' => 'Failed to fetch messages: ' . $e->getMessage(), 'code' => 500];
        }
    }

    public function sendMessage($sender_id, $receiver_id, $message) {
        try {
            // Validate that both users exist
            $userStmt = $this->pdo->prepare('SELECT id FROM users WHERE id IN (?, ?)');
            $userStmt->execute([$sender_id, $receiver_id]);
            $users = $userStmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (count($users) !== 2) {
                return ['error' => 'One or both users not found', 'code' => 404];
            }

            // Insert the message
            $stmt = $this->pdo->prepare('
                INSERT INTO messages (sender_id, receiver_id, message, created_at) 
                VALUES (?, ?, ?, NOW())
            ');
            $stmt->execute([$sender_id, $receiver_id, $message]);
            
            $message_id = $this->pdo->lastInsertId();

            // Fetch the created message with user names
            $fetchStmt = $this->pdo->prepare('
                SELECT
                    m.id,
                    m.sender_id,
                    m.receiver_id,
                    m.message,
                    m.created_at,
                    sender.full_name as sender_name,
                    receiver.full_name as receiver_name
                FROM messages m
                LEFT JOIN users sender ON m.sender_id = sender.id
                LEFT JOIN users receiver ON m.receiver_id = receiver.id
                WHERE m.id = ?
            ');
            $fetchStmt->execute([$message_id]);
            $createdMessage = $fetchStmt->fetch(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'data' => $createdMessage,
                'message' => 'Message sent successfully'
            ];
        } catch (Exception $e) {
            return ['error' => 'Failed to send message: ' . $e->getMessage(), 'code' => 500];
        }
    }

    public function getConversations($user_id) {
        try {
            $stmt = $this->pdo->prepare('
                SELECT DISTINCT
                    CASE 
                        WHEN m.sender_id = ? THEN m.receiver_id
                        ELSE m.sender_id
                    END as other_user_id,
                    CASE 
                        WHEN m.sender_id = ? THEN receiver.full_name
                        ELSE sender.full_name
                    END as other_user_name,
                    CASE 
                        WHEN m.sender_id = ? THEN receiver.email
                        ELSE sender.email
                    END as other_user_email,
                    MAX(m.created_at) as last_message_time,
                    COUNT(m.id) as message_count
                FROM messages m
                LEFT JOIN users sender ON m.sender_id = sender.id
                LEFT JOIN users receiver ON m.receiver_id = receiver.id
                WHERE m.sender_id = ? OR m.receiver_id = ?
                GROUP BY other_user_id, other_user_name, other_user_email
                ORDER BY last_message_time DESC
            ');
            $stmt->execute([$user_id, $user_id, $user_id, $user_id, $user_id]);
            $conversations = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'data' => $conversations
            ];
        } catch (Exception $e) {
            return ['error' => 'Failed to fetch conversations: ' . $e->getMessage(), 'code' => 500];
        }
    }
} 