<?php
require_once __DIR__ . '/../utils/db.php';

class ManagerController {
    private $pdo;

    public function __construct() {
        require_once __DIR__ . '/../utils/db.php';
        $this->pdo = getPDO();
    }

    public function getPDO() {
        return $this->pdo;
    }

    public function getProfile($user_id) {
        $stmt = $this->pdo->prepare('
            SELECT m.*, u.funded_balance, u.demo_balance, u.email, u.full_name, u.avatar_url
            FROM managers m
            JOIN users u ON m.user_id = u.id
            WHERE m.user_id = ?
        ');
        $stmt->execute([$user_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getApprovals($manager_id) {
        $stmt = $this->pdo->prepare('SELECT * FROM manager_approvals WHERE manager_id = ?');
        $stmt->execute([$manager_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function approveUser($manager_id, $user_id, $status) {
        $stmt = $this->pdo->prepare('UPDATE manager_approvals SET status = ?, decided_at = NOW() WHERE manager_id = ? AND user_id = ?');
        $stmt->execute([$status, $manager_id, $user_id]);
        return ['success' => true];
    }

    public function getManagedUsers($manager_id) {
        try {
            $stmt = $this->pdo->prepare('
                SELECT 
                    u.id as user_id,
                    u.full_name,
                    u.email,
                    MAX(i.created_at) as last_investment_at,
                    COUNT(i.id) as investments_count,
                    SUM(i.amount) as total_invested,
                    SUM(COALESCE(i.pnl,0)) as total_pnl
                FROM investments i
                JOIN users u ON i.user_id = u.id
                WHERE i.manager_id = ?
                GROUP BY u.id, u.full_name, u.email
                ORDER BY last_investment_at DESC
            ');
            $stmt->execute([$manager_id]);
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            return array_map(function($row){
                return [
                    'id' => intval($row['user_id']),
                    'full_name' => $row['full_name'],
                    'email' => $row['email'],
                    'last_investment_at' => $row['last_investment_at'],
                    'investments_count' => intval($row['investments_count'] ?? 0),
                    'total_invested' => floatval($row['total_invested'] ?? 0),
                    'total_pnl' => floatval($row['total_pnl'] ?? 0)
                ];
            }, $users);

        } catch (PDOException $e) {
            error_log("Error in getManagedUsers: " . $e->getMessage());
            return ['error' => 'Failed to fetch managed users', 'code' => 500];
        }
    }

    public function getNotifications($manager_id) {
        $stmt = $this->pdo->prepare('SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC');
        $stmt->execute([$manager_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function sendMessage($from, $to, $content) {
        $stmt = $this->pdo->prepare('INSERT INTO messages (sender_id, receiver_id, content) VALUES (?, ?, ?)');
        $stmt->execute([$from, $to, $content]);
        return ['success' => true];
    }

    public function getMessages($user_id, $other_id) {
        $stmt = $this->pdo->prepare('SELECT * FROM messages WHERE (sender_id = ? AND receiver_id = ?) OR (sender_id = ? AND receiver_id = ?) ORDER BY sent_at ASC');
        $stmt->execute([$user_id, $other_id, $other_id, $user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getActivityLog($manager_id) {
        try {
            $stmt = $this->pdo->prepare('SELECT * FROM activity_log WHERE manager_id = ? ORDER BY created_at DESC');
            $stmt->execute([$manager_id]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            return ['error' => 'DB error: ' . $e->getMessage(), 'code' => 500];
        }
    }

    // Return all brokers (public)
    public function getAllBrokers() {
        try {
            // Only get managers whose applications are approved
            $stmt = $this->pdo->prepare('
                SELECT m.*, u.full_name, u.email, u.avatar_url, ba.status as application_status 
                FROM managers m 
                JOIN users u ON m.user_id = u.id
                JOIN broker_applications ba ON m.user_id = ba.user_id
                WHERE ba.status = "approved"
                ORDER BY m.created_at DESC
            ');
            $stmt->execute();
            $brokers = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Just return the (possibly empty) array
            return $brokers;
        } catch (PDOException $e) {
            error_log("Error in getAllBrokers: " . $e->getMessage());
            return [];
        }
    }

    // Return broker detail and investments (public)
    public function getBrokerDetail($id, $user_id = null) {
        $stmt = $this->pdo->prepare('SELECT id, name, specialty, aum, rating, clients, image, bio FROM managers WHERE id = ?');
        $stmt->execute([$id]);
        $broker = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$broker) return ['error' => 'Not found', 'code' => 404];

        $sql = 'SELECT * FROM investments WHERE manager_id = ? AND status IN ("pending", "active")';
        $params = [$id];
        if ($user_id) {
            $sql .= ' AND user_id = ?';
            $params[] = $user_id;
        }
        $sql .= ' ORDER BY created_at DESC';
        
        $invStmt = $this->pdo->prepare($sql);
        $invStmt->execute($params);
        $investments = $invStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // For each investment, fetch its PnL history
        foreach ($investments as &$investment) {
            $pnlStmt = $this->pdo->prepare('SELECT pnl_amount as pnl, created_at as date FROM pnl_history WHERE investment_id = ? ORDER BY created_at ASC');
            $pnlStmt->execute([$investment['id']]);
            $investment['pnl_history'] = $pnlStmt->fetchAll(PDO::FETCH_ASSOC);
        }
        
        return ['broker' => $broker, 'investments' => $investments];
    }

    // User applies to be a broker
    public function applyToBeBroker($user_id, $phone = null, $experience = null, $credentials = null, $bio = null, $certification = null) {
        try {
            $this->pdo->beginTransaction();
            
            // Check if user already has a pending or approved application
            $stmt = $this->pdo->prepare('SELECT status FROM broker_applications WHERE user_id = ? ORDER BY applied_at DESC LIMIT 1');
            $stmt->execute([$user_id]);
            $existingApp = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingApp) {
                if ($existingApp['status'] === 'approved') {
                    return ['error' => 'You are already an approved manager', 'code' => 400];
                } else if ($existingApp['status'] === 'pending') {
                    return ['error' => 'You already have a pending application', 'code' => 400];
                }
            }
            
            // Insert application into broker_applications table
            $stmt = $this->pdo->prepare('INSERT INTO broker_applications (user_id, phone, experience, credentials, bio, certification, status, applied_at) VALUES (?, ?, ?, ?, ?, ?, "pending", NOW())');
            $stmt->execute([$user_id, $phone, $experience, $credentials, $bio, $certification]);
            
            // Get user details for managers table
            $stmt = $this->pdo->prepare('SELECT full_name FROM users WHERE id = ?');
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Check if user already exists in managers table
            $stmt = $this->pdo->prepare('SELECT id FROM managers WHERE user_id = ?');
            $stmt->execute([$user_id]);
            $existingManager = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingManager) {
                // Update existing manager record
                $stmt = $this->pdo->prepare('UPDATE managers SET name = ?, bio = ?, updated_at = NOW() WHERE user_id = ?');
                $stmt->execute([$user['full_name'], $bio, $user_id]);
            } else {
                // Insert new manager record
                $stmt = $this->pdo->prepare('INSERT INTO managers (user_id, name, bio, created_at, updated_at) VALUES (?, ?, ?, NOW(), NOW())');
                $stmt->execute([$user_id, $user['full_name'], $bio]);
            }
            
            $this->pdo->commit();
            return ['success' => true, 'message' => 'Application submitted successfully'];
            
        } catch (PDOException $e) {
            $this->pdo->rollBack();
            error_log("Error in applyToBeBroker: " . $e->getMessage());
            return ['error' => 'Failed to submit application: ' . $e->getMessage(), 'code' => 500];
        }
    }

    public function getBrokerApplicationStatus($user_id) {
        $stmt = $this->pdo->prepare('SELECT * FROM broker_applications WHERE user_id = ? ORDER BY applied_at DESC LIMIT 1');
        $stmt->execute([$user_id]);
        $app = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$app) return ['status' => 'none'];
        return ['status' => $app['status'], 'application' => $app];
    }

    // List all investments for this manager
    public function getManagerInvestments($manager_user_id) {
        // Resolve manager id
        $stmt = $this->pdo->prepare('SELECT id FROM managers WHERE user_id = ?');
        $stmt->execute([$manager_user_id]);
        $mgr = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$mgr) return ['error' => 'Manager not found', 'code' => 404];
        $manager_id = intval($mgr['id']);

        $q = $this->pdo->prepare('SELECT i.id, i.user_id, u.full_name AS user_name, u.email AS user_email, i.amount, i.account_type, i.status, i.created_at, COALESCE(i.pnl,0) AS pnl
                                   FROM investments i JOIN users u ON u.id = i.user_id
                                   WHERE i.manager_id = ? ORDER BY i.created_at DESC');
        $q->execute([$manager_id]);
        return ['investments' => $q->fetchAll(PDO::FETCH_ASSOC)];
    }

    // List all investments for a specific user under this manager
    public function getUserInvestmentsForManager($manager_user_id, $target_user_id) {
        $stmt = $this->pdo->prepare('SELECT id FROM managers WHERE user_id = ?');
        $stmt->execute([$manager_user_id]);
        $mgr = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$mgr) return ['error' => 'Manager not found', 'code' => 404];
        $manager_id = intval($mgr['id']);

        $q = $this->pdo->prepare('SELECT i.id, i.amount, i.account_type, i.status, i.created_at, COALESCE(i.pnl,0) AS pnl
                                   FROM investments i WHERE i.manager_id = ? AND i.user_id = ? ORDER BY i.created_at DESC');
        $q->execute([$manager_id, $target_user_id]);
        return ['investments' => $q->fetchAll(PDO::FETCH_ASSOC)];
    }

    // Pause or resume demo transactions for a specific user (by manager)
    public function setUserDemoPaused($manager_user_id, $target_user_id, $paused, $reason = null, $expires_at = null) {
        // Resolve manager id from managers table
        $stmt = $this->pdo->prepare('SELECT id FROM managers WHERE user_id = ?');
        $stmt->execute([$manager_user_id]);
        $manager = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$manager) return ['error' => 'Manager not found', 'code' => 404];
        $manager_id = intval($manager['id']);

        // Upsert into user_restrictions
        $stmt = $this->pdo->prepare('SELECT id FROM user_restrictions WHERE user_id = ?');
        $stmt->execute([$target_user_id]);
        $existing = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($existing) {
            $upd = $this->pdo->prepare('UPDATE user_restrictions SET demo_paused = ?, reason = ?, expires_at = ?, manager_id = ? WHERE id = ?');
            $upd->execute([ $paused ? 1 : 0, $reason, $expires_at, $manager_id, $existing['id'] ]);
        } else {
            $ins = $this->pdo->prepare('INSERT INTO user_restrictions (user_id, manager_id, demo_paused, reason, expires_at) VALUES (?, ?, ?, ?, ?)');
            $ins->execute([$target_user_id, $manager_id, $paused ? 1 : 0, $reason, $expires_at]);
        }
        return ['success' => true];
    }

    // Admin: Approve/reject manager application
    public function approveManagerApplication($application_id, $status, $admin_id) {
        try {
            $this->pdo->beginTransaction();

            // Update broker_applications table
            $stmt = $this->pdo->prepare('
                UPDATE broker_applications 
                SET status = ?, 
                    reviewed_at = NOW(), 
                    reviewer_id = ? 
                WHERE id = ?
            ');
            $stmt->execute([$status, $admin_id, $application_id]);

            // Get user_id from the application
            $stmt = $this->pdo->prepare('SELECT user_id FROM broker_applications WHERE id = ?');
            $stmt->execute([$application_id]);
            $application = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($status === 'approved') {
                // Update user role to manager if approved
                $stmt = $this->pdo->prepare('UPDATE users SET role = "manager" WHERE id = ?');
                $stmt->execute([$application['user_id']]);
            }

            $this->pdo->commit();
            return ['success' => true, 'message' => 'Application ' . $status];

        } catch (PDOException $e) {
            $this->pdo->rollBack();
            error_log("Error in approveManagerApplication: " . $e->getMessage());
            return ['error' => 'Failed to update application status: ' . $e->getMessage(), 'code' => 500];
        }
    }

    public function getManagerStats($manager_id) {
        // Total AUM (active investments sum)
        $stmt_invested = $this->pdo->prepare('SELECT SUM(amount) as total_invested FROM investments WHERE manager_id = ? AND status = "active"');
        $stmt_invested->execute([$manager_id]);
        $totalInvested = floatval($stmt_invested->fetch(PDO::FETCH_ASSOC)['total_invested'] ?? 0);

        // Active clients = distinct users with at least one active investment
        $stmt_active_clients = $this->pdo->prepare('SELECT COUNT(DISTINCT user_id) as cnt FROM investments WHERE manager_id = ? AND status = "active"');
        $stmt_active_clients->execute([$manager_id]);
        $activeClients = intval($stmt_active_clients->fetch(PDO::FETCH_ASSOC)['cnt'] ?? 0);

        // Pending approvals = investments in pending
        $stmt_pending = $this->pdo->prepare('SELECT COUNT(*) as cnt FROM investments WHERE manager_id = ? AND status = "pending"');
        $stmt_pending->execute([$manager_id]);
        $pendingApprovals = intval($stmt_pending->fetch(PDO::FETCH_ASSOC)['cnt'] ?? 0);

        // Total P&L from active investments
        $stmt_pnl = $this->pdo->prepare('SELECT COALESCE(SUM(pnl),0) as total_pnl FROM investments WHERE manager_id = ? AND status = "active"');
        $stmt_pnl->execute([$manager_id]);
        $totalPnl = floatval($stmt_pnl->fetch(PDO::FETCH_ASSOC)['total_pnl'] ?? 0);

        return [
            'total_aum' => $totalInvested,
            'active_clients' => $activeClients,
            'pending_approvals' => $pendingApprovals,
            'total_pnl' => $totalPnl,
        ];
    }

    public function updateProfile($manager_id, $data) {
        $allowedFields = [
            'name', 'bio', 'specialty', 'experience_years', 'avg_return', 'rating',
            'trading_specialties', 'trading_style', 'risk_tolerance', 
            'years_forex', 'years_crypto', 'years_stocks', 
            'preferred_session', 'max_drawdown',
            'allow_withdrawals', 'allow_cancellations', 
            'withdrawal_message', 'cancellation_message'
        ];
        
        $updateFields = [];
        $updateValues = [];
        
        foreach ($data as $key => $value) {
            if (in_array($key, $allowedFields)) {
                $updateFields[] = "$key = ?";
                $updateValues[] = $value;
            }
        }
        
        if (empty($updateFields)) {
            return ['error' => 'No valid fields to update', 'code' => 400];
        }
        
        $updateValues[] = $manager_id;
        $sql = "UPDATE managers SET " . implode(', ', $updateFields) . " WHERE id = ?";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($updateValues);
        
        return ['success' => true, 'message' => 'Profile updated successfully'];
    }

    public function getPermissions($manager_id) {
        $stmt = $this->pdo->prepare('SELECT allow_withdrawals, allow_cancellations, withdrawal_message, cancellation_message FROM managers WHERE id = ?');
        $stmt->execute([$manager_id]);
        $permissions = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$permissions) {
            return ['error' => 'Manager not found', 'code' => 404];
        }
        
        return ['permissions' => $permissions];
    }

    public function updatePermissions($manager_id, $data) {
        $stmt = $this->pdo->prepare('UPDATE managers SET allow_withdrawals = ?, allow_cancellations = ?, withdrawal_message = ?, cancellation_message = ? WHERE id = ?');
        $stmt->execute([
            $data['allow_withdrawals'] ?? true,
            $data['allow_cancellations'] ?? true,
            $data['withdrawal_message'] ?? null,
            $data['cancellation_message'] ?? null,
            $manager_id
        ]);
        
        return ['success' => true, 'message' => 'Permissions updated successfully'];
    }

    public function createCampaign($manager_id, $data) {
        $stmt = $this->pdo->prepare('INSERT INTO investment_campaigns (manager_id, title, description, min_amount, max_amount, target_amount, trading_type, trading_pairs, risk_level) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)');
        $stmt->execute([
            $manager_id,
            $data['title'],
            $data['description'] ?? null,
            $data['min_amount'],
            $data['max_amount'] ?? null,
            $data['target_amount'] ?? null,
            $data['trading_type'] ?? null,
            $data['trading_pairs'] ?? null,
            $data['risk_level'] ?? null
        ]);
        
        return ['success' => true, 'campaign_id' => $this->pdo->lastInsertId()];
    }

    public function getCampaigns($manager_id) {
        $stmt = $this->pdo->prepare('SELECT * FROM investment_campaigns WHERE manager_id = ? ORDER BY created_at DESC');
        $stmt->execute([$manager_id]);
        $campaigns = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return ['campaigns' => $campaigns];
    }

    public function updateCampaignStatus($manager_id, $campaign_id, $status) {
        $stmt = $this->pdo->prepare('UPDATE investment_campaigns SET status = ? WHERE id = ? AND manager_id = ?');
        $stmt->execute([$status, $campaign_id, $manager_id]);
        
        if ($stmt->rowCount() === 0) {
            return ['error' => 'Campaign not found', 'code' => 404];
        }
        
        return ['success' => true, 'message' => 'Campaign status updated successfully'];
    }

    public function getInvestors($manager_id, $page = 1, $limit = 10, $search = '') {
        $offset = ($page - 1) * $limit;
        
        // Convert to integers to avoid SQL syntax errors
        $limit = (int)$limit;
        $offset = (int)$offset;
        
        // Base query to get investors with their investment summary
        $searchCondition = '';
        $searchParams = [$manager_id];
        
        if (!empty($search)) {
            $searchCondition = ' AND (u.full_name LIKE ? OR u.email LIKE ?)';
            $searchParams[] = "%$search%";
            $searchParams[] = "%$search%";
        }
        
        // Get total count
        $countSql = "SELECT COUNT(DISTINCT u.id) as total 
                     FROM users u 
                     INNER JOIN investments i ON u.id = i.user_id 
                     WHERE i.manager_id = ? $searchCondition";
        $countStmt = $this->pdo->prepare($countSql);
        $countStmt->execute($searchParams);
        $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get investors with pagination
        $sql = "SELECT u.id, u.full_name as name, u.email, u.avatar_url as avatar,
                       COUNT(i.id) as active_investments,
                       SUM(i.amount) as total_invested,
                       SUM(i.pnl) as total_pnl,
                       MAX(i.created_at) as last_investment,
                       'active' as status,
                       GROUP_CONCAT(DISTINCT i.account_type) as account_type
                FROM users u 
                INNER JOIN investments i ON u.id = i.user_id 
                WHERE i.manager_id = ? $searchCondition
                GROUP BY u.id, u.full_name, u.email, u.avatar_url
                ORDER BY total_invested DESC
                LIMIT $limit OFFSET $offset";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($searchParams);
        $investors = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get detailed investments for each investor
        foreach ($investors as &$investor) {
            $investmentStmt = $this->pdo->prepare('SELECT id, amount, pnl, status, trading_type, risk_level, created_at FROM investments WHERE user_id = ? AND manager_id = ? ORDER BY created_at DESC');
            $investmentStmt->execute([$investor['id'], $manager_id]);
            $investor['investments'] = $investmentStmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Clean up account_type (get primary account type)
            $accountTypes = explode(',', $investor['account_type']);
            $investor['account_type'] = in_array('funded', $accountTypes) ? 'funded' : 'demo';
        }
        
        return [
            'investors' => $investors,
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'total_pages' => ceil($total / $limit)
        ];
    }

    public function handleBulkInvestorAction($manager_id, $data) {
        $action = $data['action'] ?? null;
        $investor_ids = $data['investor_ids'] ?? [];
        
        if (!$action || !is_array($investor_ids) || empty($investor_ids)) {
            return ['error' => 'Action and investor_ids are required', 'code' => 400];
        }
        
        $allowed_actions = ['pause', 'resume', 'notify', 'export'];
        if (!in_array($action, $allowed_actions)) {
            return ['error' => 'Invalid action. Allowed: ' . implode(', ', $allowed_actions), 'code' => 400];
        }
        
        try {
            $this->pdo->beginTransaction();
            
            $affected_count = 0;
            
            switch ($action) {
                case 'pause':
                    // Pause all active investments for these investors
                    $placeholders = str_repeat('?,', count($investor_ids) - 1) . '?';
                    $sql = "UPDATE investments SET status = 'paused' WHERE user_id IN ($placeholders) AND manager_id = ? AND status = 'active'";
                    $params = array_merge($investor_ids, [$manager_id]);
                    $stmt = $this->pdo->prepare($sql);
                    $stmt->execute($params);
                    $affected_count = $stmt->rowCount();
                    break;
                    
                case 'resume':
                    // Resume all paused investments for these investors
                    $placeholders = str_repeat('?,', count($investor_ids) - 1) . '?';
                    $sql = "UPDATE investments SET status = 'active' WHERE user_id IN ($placeholders) AND manager_id = ? AND status = 'paused'";
                    $params = array_merge($investor_ids, [$manager_id]);
                    $stmt = $this->pdo->prepare($sql);
                    $stmt->execute($params);
                    $affected_count = $stmt->rowCount();
                    break;
                    
                case 'notify':
                    // Create notifications for these investors
                    $message = $data['message'] ?? 'You have a new notification from your manager.';
                    foreach ($investor_ids as $investor_id) {
                        $stmt = $this->pdo->prepare('INSERT INTO notifications (user_id, title, message, type) VALUES (?, ?, ?, ?)');
                        $stmt->execute([$investor_id, 'Manager Notification', $message, 'manager_message']);
                        $affected_count++;
                    }
                    break;
                    
                case 'export':
                    // For export, we'll just return the investor data
                    $placeholders = str_repeat('?,', count($investor_ids) - 1) . '?';
                    $sql = "SELECT u.id, u.full_name, u.email, 
                                   COUNT(i.id) as investment_count,
                                   SUM(i.amount) as total_invested,
                                   SUM(i.pnl) as total_pnl
                            FROM users u 
                            INNER JOIN investments i ON u.id = i.user_id 
                            WHERE u.id IN ($placeholders) AND i.manager_id = ?
                            GROUP BY u.id, u.full_name, u.email";
                    $params = array_merge($investor_ids, [$manager_id]);
                    $stmt = $this->pdo->prepare($sql);
                    $stmt->execute($params);
                    $export_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    $affected_count = count($export_data);
                    
                    $this->pdo->commit();
                    return [
                        'success' => true,
                        'action' => $action,
                        'affected_count' => $affected_count,
                        'export_data' => $export_data
                    ];
            }
            
            $this->pdo->commit();
            return [
                'success' => true,
                'action' => $action,
                'affected_count' => $affected_count,
                'message' => ucfirst($action) . " action completed successfully for $affected_count items"
            ];
            
        } catch (PDOException $e) {
            $this->pdo->rollBack();
            error_log("Error in handleBulkInvestorAction: " . $e->getMessage());
            return ['error' => 'Failed to perform bulk action: ' . $e->getMessage(), 'code' => 500];
        }
    }
} 