<?php

require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../utils/jwt.php';

class JointAccountController {
    private $pdo;

    public function __construct() {
        $this->pdo = getPDO();
        require_once __DIR__ . '/../utils/mail.php';
    }

    // Create a joint account invitation
    public function createInvitation($user_id, $data) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            $invitee_email = $data['invitee_email'] ?? '';
            $invitee_name = $data['invitee_name'] ?? '';
            $account_type = $data['account_type'] ?? 'funded';
            $permissions = $data['permissions'] ?? ['view', 'trade', 'withdraw'];

            if (empty($invitee_email)) {
                return ['error' => 'Invitee email is required'];
            }

            // Check if user is trying to invite themselves
            $stmt = $this->pdo->prepare("SELECT email FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $current_user = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($current_user) && strtolower(trim($current_user[0]['email'])) === strtolower(trim($invitee_email))) {
                return ['error' => 'You cannot invite yourself to a joint account'];
            }

            // Check if invitee already exists and has a joint account with this user
            $stmt = $this->pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$invitee_email]);
            $invitee_user = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($invitee_user)) {
                $invitee_user_id = $invitee_user[0]['id'];
                
                // Check if they already have a joint account together
                $stmt = $this->pdo->prepare("SELECT ja.id FROM joint_accounts ja 
                     JOIN joint_account_members jam1 ON ja.id = jam1.joint_account_id 
                     JOIN joint_account_members jam2 ON ja.id = jam2.joint_account_id 
                     WHERE jam1.user_id = ? AND jam2.user_id = ? AND ja.account_type = ? AND ja.status = 'active'");
                $stmt->execute([$user_id, $invitee_user_id, $account_type]);
                $existing_joint_account = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (!empty($existing_joint_account)) {
                    return ['error' => 'You already have a joint account with this user'];
                }
            }

            // Check if there's already a pending invitation to this email
            $stmt = $this->pdo->prepare("SELECT id FROM joint_account_invitations 
                 WHERE inviter_id = ? AND invitee_email = ? AND status = 'pending' AND expires_at > NOW()");
            $stmt->execute([$user_id, $invitee_email]);
            $pending_invitation = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($pending_invitation)) {
                return ['error' => 'You already have a pending invitation to this email address'];
            }

            // Check if user already has a joint account
            $stmt = $this->pdo->prepare("SELECT jam.joint_account_id FROM joint_account_members jam 
                 JOIN joint_accounts ja ON jam.joint_account_id = ja.id 
                 WHERE jam.user_id = ? AND ja.account_type = ? AND ja.status = 'active'");
            $stmt->execute([$user_id, $account_type]);
            $existing_account = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (!empty($existing_account)) {
                return ['error' => 'You already have an active joint account of this type'];
            }

            // Generate invitation token
            $invitation_token = bin2hex(random_bytes(32));
            $expires_at = date('Y-m-d H:i:s', strtotime('+7 days'));

            // Create invitation
            $stmt = $this->pdo->prepare("INSERT INTO joint_account_invitations (inviter_id, invitee_email, invitee_name, account_type, permissions, invitation_token, expires_at) 
                 VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$user_id, $invitee_email, $invitee_name, $account_type, json_encode($permissions), $invitation_token, $expires_at]);
            $invitation_id = $this->pdo->lastInsertId();

            if ($invitation_id) {
                // Get inviter details for email
                $stmt = $this->pdo->prepare("SELECT full_name FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $inviter = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $inviter_name = $inviter[0]['full_name'] ?? 'User';
                
                // Send invitation email
                $email_sent = sendJointAccountInvitation(
                    $invitee_email,
                    $invitee_name ?: 'there',
                    $inviter_name,
                    $invitation_token,
                    $account_type,
                    $permissions
                );
                
                // Log activity
                $this->logActivity($user_id, 'created_invitation', "Invited {$invitee_email} to joint account");
                
                return [
                    'success' => true,
                    'message' => 'Invitation sent successfully' . ($email_sent ? '' : ' (email delivery may be delayed)'),
                    'invitation_id' => $invitation_id,
                    'invitation_token' => $invitation_token,
                    'email_sent' => $email_sent
                ];
            }

            return ['error' => 'Failed to create invitation'];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Validate joint account invitation
    public function validateInvitation($token) {
        try {
            if (empty($token)) {
                return ['error' => 'Token is required'];
            }

            // Get invitation details
            $stmt = $this->pdo->prepare("SELECT jai.*, u.full_name as inviter_name 
                 FROM joint_account_invitations jai 
                 JOIN users u ON jai.inviter_id = u.id 
                 WHERE jai.invitation_token = ? AND jai.status = 'pending' AND jai.expires_at > NOW()");
            $stmt->execute([$token]);
            $invitation = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($invitation)) {
                return ['error' => 'Invalid or expired invitation'];
            }

            $invitation = $invitation[0];

            return [
                'success' => true,
                'invitation' => [
                    'inviter_name' => $invitation['inviter_name'],
                    'account_type' => $invitation['account_type'],
                    'permissions' => json_decode($invitation['permissions'], true),
                    'expires_at' => $invitation['expires_at']
                ]
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Accept joint account invitation
    public function acceptInvitation($user_id, $token) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Get invitation details
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_invitations WHERE invitation_token = ? AND status = 'pending' AND expires_at > NOW()");
            $stmt->execute([$token]);
            $invitation = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($invitation)) {
                return ['error' => 'Invalid or expired invitation'];
            }

            $invitation = $invitation[0];

            // Check if user already has a joint account of this type
            $stmt = $this->pdo->prepare("SELECT jam.joint_account_id FROM joint_account_members jam 
                 JOIN joint_accounts ja ON jam.joint_account_id = ja.id 
                 WHERE jam.user_id = ? AND ja.account_type = ? AND ja.status = 'active'");
            $stmt->execute([$user_id, $invitation['account_type']]);
            $existing_account = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (!empty($existing_account)) {
                return ['error' => 'You already have an active joint account of this type'];
            }

            // Check if a joint account already exists between these two users
            $stmt = $this->pdo->prepare("SELECT ja.id FROM joint_accounts ja 
                 JOIN joint_account_members jam1 ON ja.id = jam1.joint_account_id 
                 JOIN joint_account_members jam2 ON ja.id = jam2.joint_account_id 
                 WHERE jam1.user_id = ? AND jam2.user_id = ? AND ja.account_type = ? AND ja.status = 'active'");
            $stmt->execute([$invitation['inviter_id'], $user_id, $invitation['account_type']]);
            $existing_joint_account = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $joint_account_id = null;

            // Start transaction
            $this->pdo->beginTransaction();

            try {
                if (!empty($existing_joint_account)) {
                    // Use existing joint account
                    $joint_account_id = $existing_joint_account[0]['id'];
                    
                    // Check if user is already a member of this account
                    $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ?");
                    $stmt->execute([$joint_account_id, $user_id]);
                    $existing_member = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    if (!empty($existing_member)) {
                        // User is already a member, just update invitation status
                        $stmt = $this->pdo->prepare("UPDATE joint_account_invitations SET status = 'accepted', updated_at = NOW() WHERE id = ?");
                        $stmt->execute([$invitation['id']]);
                        
                        $this->pdo->commit();
                        
                        return [
                            'success' => true,
                            'message' => 'Already a member of this joint account',
                            'joint_account_id' => $joint_account_id
                        ];
                    }
                } else {
                    // Create new joint account
                    $stmt = $this->pdo->prepare("INSERT INTO joint_accounts (account_number, account_type, balance) VALUES (?, ?, 0.00)");
                    $stmt->execute([$this->generateAccountNumber(), $invitation['account_type']]);
                    $joint_account_id = $this->pdo->lastInsertId();

                    // Add inviter as primary member
                    $stmt = $this->pdo->prepare("INSERT INTO joint_account_members (joint_account_id, user_id, role, permissions) VALUES (?, ?, 'primary', ?)");
                    $stmt->execute([$joint_account_id, $invitation['inviter_id'], json_encode(['view', 'trade', 'withdraw', 'manage'])]);
                }

                // Add invitee as secondary member (only if not already a member)
                $stmt = $this->pdo->prepare("INSERT INTO joint_account_members (joint_account_id, user_id, role, permissions) VALUES (?, ?, 'secondary', ?)");
                $stmt->execute([$joint_account_id, $user_id, $invitation['permissions']]);

                // Update invitation status
                $stmt = $this->pdo->prepare("UPDATE joint_account_invitations SET status = 'accepted', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$invitation['id']]);

                // Get user details for email notification
                $stmt = $this->pdo->prepare("SELECT full_name FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $invitee = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $invitee_name = $invitee[0]['full_name'] ?? 'User';
                
                $stmt = $this->pdo->prepare("SELECT u.full_name, u.email FROM users u WHERE u.id = ?");
                $stmt->execute([$invitation['inviter_id']]);
                $inviter = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $inviter_name = $inviter[0]['full_name'] ?? 'User';
                $inviter_email = $inviter[0]['email'] ?? '';
                
                // Send acceptance notification email to inviter
                if ($inviter_email) {
                    sendJointAccountAcceptanceNotification(
                        $inviter_email,
                        $inviter_name,
                        $invitee_name,
                        $joint_account_id
                    );
                }
                
                // Log activity
                $this->logActivity($user_id, 'accepted_invitation', "Joined joint account #{$joint_account_id}");

                $this->pdo->commit();

                return [
                    'success' => true,
                    'message' => 'Joint account created successfully',
                    'joint_account_id' => $joint_account_id
                ];

            } catch (Exception $e) {
                $this->pdo->rollback();
                throw $e;
            }

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Get user's joint accounts
    public function getUserJointAccounts($user_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            $stmt = $this->pdo->prepare("SELECT ja.*, jam.role, jam.permissions, 
                        (SELECT COUNT(*) FROM joint_account_members WHERE joint_account_id = ja.id AND is_active = 1) as member_count
                 FROM joint_accounts ja
                 JOIN joint_account_members jam ON ja.id = jam.joint_account_id
                 WHERE jam.user_id = ? AND jam.is_active = 1 AND ja.status = 'active'
                 ORDER BY ja.created_at DESC");
            $stmt->execute([$user_id]);
            $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'accounts' => $accounts
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Get joint account details
    public function getJointAccountDetails($user_id, $account_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Check if user has access to this account
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            // Get account details
            $stmt = $this->pdo->prepare("SELECT * FROM joint_accounts WHERE id = ?");
            $stmt->execute([$account_id]);
            $account = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($account)) {
                return ['error' => 'Account not found'];
            }

            // Get members
            $stmt = $this->pdo->prepare("SELECT jam.*, u.full_name, u.email, u.avatar_url
                 FROM joint_account_members jam
                 JOIN users u ON jam.user_id = u.id
                 WHERE jam.joint_account_id = ? AND jam.is_active = 1
                 ORDER BY jam.role DESC, jam.joined_at ASC");
            $stmt->execute([$account_id]);
            $members = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Get recent transactions
            $stmt = $this->pdo->prepare("SELECT jat.*, u.full_name as initiated_by_name
                 FROM joint_account_transactions jat
                 JOIN users u ON jat.initiated_by = u.id
                 WHERE jat.joint_account_id = ?
                 ORDER BY jat.created_at DESC
                 LIMIT 10");
            $stmt->execute([$account_id]);
            $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'account' => $account[0],
                'members' => $members,
                'transactions' => $transactions
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Create joint account transaction
    public function createTransaction($user_id, $data) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            $account_id = $data['joint_account_id'] ?? 0;
            $type = $data['type'] ?? '';
            $amount = $data['amount'] ?? 0;
            $description = $data['description'] ?? '';

            if (!$account_id || !$type || !$amount) {
                return ['error' => 'Missing required fields'];
            }

            // Check if user has access and permissions
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            $permissions = json_decode($member[0]['permissions'], true);
            
            // Check if transaction requires approval
            $requires_approval = $this->requiresApproval($type, $amount, $permissions);

            $stmt = $this->pdo->prepare("INSERT INTO joint_account_transactions (joint_account_id, initiated_by, type, amount, description, requires_approval, status) 
                 VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$account_id, $user_id, $type, $amount, $description, $requires_approval ? 1 : 0, $requires_approval ? 'pending' : 'completed']);
            $transaction_id = $this->pdo->lastInsertId();

            if ($transaction_id) {
                // Log activity
                $this->logActivity($user_id, 'created_transaction', "Created {$type} transaction for \${$amount}");
                
                return [
                    'success' => true,
                    'message' => $requires_approval ? 'Transaction created, awaiting approval' : 'Transaction completed',
                    'transaction_id' => $transaction_id,
                    'requires_approval' => $requires_approval
                ];
            }

            return ['error' => 'Failed to create transaction'];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Approve joint account transaction
    public function approveTransaction($user_id, $transaction_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Get transaction details
            $stmt = $this->pdo->prepare("SELECT jat.*, ja.account_type, ja.balance
                 FROM joint_account_transactions jat
                 JOIN joint_accounts ja ON jat.joint_account_id = ja.id
                 WHERE jat.id = ? AND jat.status = 'pending'");
            $stmt->execute([$transaction_id]);
            $transaction = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($transaction)) {
                return ['error' => 'Transaction not found or already processed'];
            }

            $transaction = $transaction[0];

            // Check if user can approve (primary member or has approval permissions)
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$transaction['joint_account_id'], $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            $permissions = json_decode($member[0]['permissions'], true);
            if ($member[0]['role'] !== 'primary' && !in_array('approve', $permissions)) {
                return ['error' => 'Insufficient permissions to approve transactions'];
            }

            // Start transaction
            $this->pdo->beginTransaction();

            try {
                // Update transaction status
                $stmt = $this->pdo->prepare("UPDATE joint_account_transactions SET status = 'completed', approved_by = ?, approved_at = NOW() WHERE id = ?");
                $stmt->execute([$user_id, $transaction_id]);

                // Update account balance
                $new_balance = $this->calculateNewBalance($transaction['type'], $transaction['amount'], $transaction['balance']);
                $stmt = $this->pdo->prepare("UPDATE joint_accounts SET balance = ? WHERE id = ?");
                $stmt->execute([$new_balance, $transaction['joint_account_id']]);

                // Log activity
                $this->logActivity($user_id, 'approved_transaction', "Approved transaction #{$transaction_id}");

                $this->pdo->commit();

                return [
                    'success' => true,
                    'message' => 'Transaction approved successfully'
                ];

            } catch (Exception $e) {
                $this->pdo->rollback();
                throw $e;
            }

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Get pending transactions for approval
    public function getPendingTransactions($user_id, $account_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Check if user has approval permissions
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            $permissions = json_decode($member[0]['permissions'], true);
            if ($member[0]['role'] !== 'primary' && !in_array('approve', $permissions)) {
                return ['error' => 'Insufficient permissions'];
            }

            $stmt = $this->pdo->prepare("SELECT jat.*, u.full_name as initiated_by_name
                 FROM joint_account_transactions jat
                 JOIN users u ON jat.initiated_by = u.id
                 WHERE jat.joint_account_id = ? AND jat.status = 'pending' AND jat.requires_approval = 1
                 ORDER BY jat.created_at ASC");
            $stmt->execute([$account_id]);
            $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'transactions' => $transactions
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Remove member from joint account
    public function removeMember($user_id, $account_id, $member_user_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Check if user is primary member
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member) || $member[0]['role'] !== 'primary') {
                return ['error' => 'Only primary members can remove other members'];
            }

            // Cannot remove primary member
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $member_user_id]);
            $target_member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($target_member)) {
                return ['error' => 'Member not found'];
            }

            if ($target_member[0]['role'] === 'primary') {
                return ['error' => 'Cannot remove primary member'];
            }

            // Deactivate member
            $stmt = $this->pdo->prepare("UPDATE joint_account_members SET is_active = 0, updated_at = NOW() WHERE joint_account_id = ? AND user_id = ?");
            $stmt->execute([$account_id, $member_user_id]);

            // Log activity
            $this->logActivity($user_id, 'removed_member', "Removed member {$member_user_id} from joint account");

            return [
                'success' => true,
                'message' => 'Member removed successfully'
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Helper methods
    private function generateAccountNumber() {
        return 'JA' . date('Y') . str_pad(mt_rand(1, 999999), 6, '0', STR_PAD_LEFT);
    }

    private function requiresApproval($type, $amount, $permissions) {
        // Large amounts or sensitive operations require approval
        if ($amount > 1000 || $type === 'withdrawal') {
            return true;
        }
        return false;
    }

    private function calculateNewBalance($type, $amount, $current_balance) {
        switch ($type) {
            case 'deposit':
                return $current_balance + $amount;
            case 'withdrawal':
                return $current_balance - $amount;
            default:
                return $current_balance;
        }
    }

    private function logActivity($user_id, $action, $details) {
        $stmt = $this->pdo->prepare("INSERT INTO joint_account_activity_log (user_id, action, details, ip_address) VALUES (?, ?, ?, ?)");
        $stmt->execute([$user_id, $action, $details, $_SERVER['REMOTE_ADDR'] ?? null]);
    }

    // Execute trade in joint account
    public function executeTrade($user_id, $data) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            $account_id = $data['joint_account_id'] ?? 0;
            $symbol = $data['symbol'] ?? '';
            $quantity = $data['quantity'] ?? 0;
            $order_type = $data['order_type'] ?? 'market';
            $side = $data['side'] ?? 'buy';

            if (!$account_id || !$symbol || !$quantity) {
                return ['error' => 'Missing required fields'];
            }

            // Check if user has access and trading permissions
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            $permissions = json_decode($member[0]['permissions'], true);
            if (!in_array('trade', $permissions)) {
                return ['error' => 'Trading permission required'];
            }

            // Get current market price (simplified - in real implementation, use market data API)
            $market_price = $this->getMarketPrice($symbol);
            if (!$market_price) {
                return ['error' => 'Unable to get market price for ' . $symbol];
            }

            // Calculate total cost
            $total_cost = $quantity * $market_price;

            // Check account balance
            $stmt = $this->pdo->prepare("SELECT balance FROM joint_accounts WHERE id = ?");
            $stmt->execute([$account_id]);
            $account = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($account)) {
                return ['error' => 'Account not found'];
            }

            if ($side === 'buy' && $account[0]['balance'] < $total_cost) {
                return ['error' => 'Insufficient balance'];
            }

            // Execute trade
            $this->pdo->beginTransaction();

            try {
                // Create trade record
                $stmt = $this->pdo->prepare("INSERT INTO joint_account_transactions (joint_account_id, type, symbol, quantity, price, side, order_type, initiated_by, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$account_id, 'trade', $symbol, $quantity, $market_price, $side, $order_type, $user_id, 'completed']);
                $transaction_id = $this->pdo->lastInsertId();

                // Update account balance
                if ($side === 'buy') {
                    $new_balance = $account[0]['balance'] - $total_cost;
                } else {
                    $new_balance = $account[0]['balance'] + $total_cost;
                }

                $stmt = $this->pdo->prepare("UPDATE joint_accounts SET balance = ? WHERE id = ?");
                $stmt->execute([$new_balance, $account_id]);

                // Log activity
                $this->logActivity($user_id, 'trade_executed', "Executed {$side} trade for {$quantity} {$symbol} @ {$market_price}");

                $this->pdo->commit();

                return [
                    'success' => true,
                    'message' => 'Trade executed successfully',
                    'transaction_id' => $transaction_id,
                    'trade_details' => [
                        'symbol' => $symbol,
                        'quantity' => $quantity,
                        'price' => $market_price,
                        'side' => $side,
                        'total' => $total_cost
                    ]
                ];

            } catch (Exception $e) {
                $this->pdo->rollback();
                throw $e;
            }

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Get joint account positions
    public function getPositions($user_id, $account_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Check if user has access
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            // Get positions (simplified - in real implementation, calculate from trades)
            $stmt = $this->pdo->prepare("SELECT symbol, SUM(CASE WHEN side = 'buy' THEN quantity ELSE -quantity END) as net_quantity,
                 AVG(CASE WHEN side = 'buy' THEN price ELSE NULL END) as avg_price
                 FROM joint_account_transactions 
                 WHERE joint_account_id = ? AND type = 'trade' 
                 GROUP BY symbol 
                 HAVING net_quantity > 0");
            $stmt->execute([$account_id]);
            $positions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Calculate unrealized P&L for each position
            foreach ($positions as &$position) {
                $current_price = $this->getMarketPrice($position['symbol']);
                if ($current_price) {
                    $position['current_price'] = $current_price;
                    $position['unrealized_pnl'] = ($current_price - $position['avg_price']) * $position['net_quantity'];
                    $position['unrealized_pnl_percent'] = (($current_price - $position['avg_price']) / $position['avg_price']) * 100;
                }
            }

            return [
                'success' => true,
                'positions' => $positions
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Get joint account orders
    public function getOrders($user_id, $account_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Check if user has access
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            // Get recent orders
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_transactions 
                 WHERE joint_account_id = ? AND type = 'trade' 
                 ORDER BY created_at DESC 
                 LIMIT 20");
            $stmt->execute([$account_id]);
            $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return [
                'success' => true,
                'orders' => $orders
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Get joint account performance
    public function getPerformance($user_id, $account_id) {
        try {
            if (!$user_id) {
                return ['error' => 'Unauthorized'];
            }

            // Check if user has access
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_members WHERE joint_account_id = ? AND user_id = ? AND is_active = 1");
            $stmt->execute([$account_id, $user_id]);
            $member = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($member)) {
                return ['error' => 'Access denied'];
            }

            // Calculate performance metrics (simplified)
            $stmt = $this->pdo->prepare("SELECT 
                 SUM(CASE WHEN side = 'sell' THEN (price - avg_price) * quantity ELSE 0 END) as total_pnl,
                 COUNT(*) as total_trades,
                 COUNT(CASE WHEN side = 'sell' AND (price - avg_price) > 0 THEN 1 END) as winning_trades
                 FROM joint_account_transactions 
                 WHERE joint_account_id = ? AND type = 'trade'");
            $stmt->execute([$account_id]);
            $performance = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $total_pnl = $performance[0]['total_pnl'] ?? 0;
            $total_trades = $performance[0]['total_trades'] ?? 0;
            $winning_trades = $performance[0]['winning_trades'] ?? 0;

            $win_rate = $total_trades > 0 ? ($winning_trades / $total_trades) * 100 : 0;

            return [
                'success' => true,
                'performance' => [
                    'totalPnL' => $total_pnl,
                    'dailyPnL' => $total_pnl * 0.1, // Simplified daily calculation
                    'totalReturn' => $total_pnl > 0 ? ($total_pnl / 10000) * 100 : 0, // Simplified return calculation
                    'winRate' => $win_rate
                ]
            ];

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }

    // Helper method to get market price (simplified)
    private function getMarketPrice($symbol) {
        // In a real implementation, this would call a market data API
        // For now, return mock prices
        $mock_prices = [
            'BTC' => 45000,
            'ETH' => 3000,
            'AAPL' => 150,
            'GOOGL' => 2800,
            'TSLA' => 250,
            'MSFT' => 350,
            'AMZN' => 3300,
            'NVDA' => 500,
            'META' => 300,
            'NFLX' => 400
        ];

        return $mock_prices[strtoupper($symbol)] ?? 100;
    }
}
