<?php

require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../services/MarketDataService.php';

class InvestmentController {
    private $pdo;
    private $marketData;

    public function __construct() {
        $this->pdo = getPDO();
        $this->marketData = new MarketDataService();
    }

    // --- NEW METHODS (Full Functionality) ---

    // LIST available investments (proxy to MarketDataService + local DB listings)
    public function index() {
        $type = $_GET['type'] ?? 'stock';
        $listings = $this->getListingsByType($type);
        
        // Enrich with live price
        foreach ($listings as &$item) {
            if ($type == 'stock' || $type == 'etf') {
                $priceData = $this->marketData->getStockPrice($item['symbol']);
                $item['current_price'] = $priceData['price'];
                $item['change_percent'] = $priceData['percentage'] ?? 0;
            } elseif ($type == 'crypto' || $type == 'commodity') {
                $priceData = $this->marketData->getCryptoPrice($item['api_id'] ?? $item['symbol']);
                 $item['current_price'] = $priceData['price'];
                 $item['change_percent'] = $priceData['change'] ?? 0;
            }
        }

        echo json_encode($listings);
    }

    // BUY Asset (Unified)
    public function buy($userId) {
        // Get JSON input
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!$data) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        $type = $data['type'] ?? null;
        $symbol = $data['symbol'] ?? null;
        $amount = $data['amount'] ?? null;
        $name = $data['name'] ?? $symbol;
        
        if (!$type || !$symbol || !$amount) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required fields: type, symbol, amount']);
            return;
        }

        // Check user balance
        $stmt = $this->pdo->prepare("SELECT funded_balance FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user || $user['funded_balance'] < $amount) {
            http_response_code(400);
            echo json_encode(['error' => 'Insufficient balance']);
            return;
        }

        // 1. Get Live Price
        $price = $this->fetchCurrentPrice($type, $symbol);
        if ($price <= 0) $price = 1; // Safety fallback

        $quantity = $amount / $price;

        // 2. Deduct balance
        $stmt = $this->pdo->prepare("UPDATE users SET funded_balance = funded_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $userId]);

        // 3. Save to DB
        $result = $this->createInvestment($userId, [
            'type' => $type,
            'symbol' => $symbol,
            'name' => $name,
            'amount' => $amount, // Invested amount
            'quantity' => $quantity,
            'purchase_price' => $price,
            'status' => 'active'
        ]);
        
        echo json_encode($result);
    }

    // --- LEGACY / HELPER METHODS (Handler Support) ---

    public function createInvestment($user_id, $data) {
        $type = $data['type'] ?? 'other';
        $symbol = $data['symbol'] ?? 'UNKNOWN';
        $name = $data['name'] ?? 'Investment';
        $amount = $data['amount'] ?? 0;
        $quantity = $data['quantity'] ?? 1;
        $purchase_price = $data['purchase_price'] ?? $amount;
        $status = $data['status'] ?? 'pending';

        $stmt = $this->pdo->prepare("
            INSERT INTO investments (user_id, type, symbol, name, quantity, purchase_price, current_value, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        try {
            $stmt->execute([$user_id, $type, $symbol, $name, $quantity, $purchase_price, $amount, $status]);
            $id = $this->pdo->lastInsertId();
            
            // Record Transaction
            $this->recordTransaction($user_id, 'buy_asset', $amount, "Bought $name ($symbol)");

            return ['success' => true, 'id' => $id, 'message' => "Investment created"];
        } catch (Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }

    public function getUserInvestments($user_id) {
        $stmt = $this->pdo->prepare("SELECT * FROM investments WHERE user_id = ? ORDER BY purchase_date DESC");
        $stmt->execute([$user_id]);
        $investments = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Update with live values on read? Optional optimization
        return $investments;
    }

    public function getPortfolio($userId, $type = null) {
        $sql = "SELECT * FROM investments WHERE user_id = ? AND status = 'active'";
        $params = [$userId];
        
        if ($type) {
            $sql .= " AND type = ?";
            $params[] = $type;
        }
        
        $sql .= " ORDER BY purchase_date DESC";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $investments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Enrich with current prices and P&L
        foreach ($investments as &$inv) {
            $currentPrice = $this->fetchCurrentPrice($inv['type'], $inv['symbol']);
            $inv['current_price'] = $currentPrice;
            $inv['current_value'] = $currentPrice * $inv['quantity'];
            $inv['profit_loss'] = $inv['current_value'] - ($inv['purchase_price'] * $inv['quantity']);
            $inv['profit_loss_percent'] = (($currentPrice - $inv['purchase_price']) / $inv['purchase_price']) * 100;
        }
        
        echo json_encode($investments);
    }

    public function getInvestmentDetails($id) {
        $stmt = $this->pdo->prepare("SELECT * FROM investments WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function updateInvestmentStatus($user_id, $investment_id, $status, $type=null, $pairs=null, $risk=null) {
        $stmt = $this->pdo->prepare("UPDATE investments SET status = ? WHERE id = ?");
        $stmt->execute([$status, $investment_id]);
        return ['success' => true];
    }

    public function addPnL($manager_id, $investment_id, $amount) {
        // Find investment
        $stmt = $this->pdo->prepare("SELECT current_value FROM investments WHERE id = ?");
        $stmt->execute([$investment_id]);
        $inv = $stmt->fetch();
        
        if ($inv) {
            $newVal = $inv['current_value'] + $amount;
            $uStmt = $this->pdo->prepare("UPDATE investments SET current_value = ? WHERE id = ?");
            $uStmt->execute([$newVal, $investment_id]);
            
            // Record usage
            $this->recordTransaction($manager_id, 'pnl_update', $amount, "PnL Update for Investment #$investment_id");
            
            return ['success' => true, 'new_value' => $newVal];
        }
        return ['error' => 'Investment not found'];
    }
    
    public function updateInvestment($id, $data) {
        // Generic update
        // Build query dynamically
        $fields = [];
        $params = [];
        foreach ($data as $k => $v) {
            $fields[] = "$k = ?";
            $params[] = $v;
        }
        $params[] = $id;
        
        if (empty($fields)) return ['success' => true];
        
        $sql = "UPDATE investments SET " . implode(', ', $fields) . " WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        
        return ['success' => true];
    }

    public function getPnlHistory($id) {
        // Mock history if table doesn't exist, or query transactions
        $stmt = $this->pdo->prepare("SELECT * FROM transactions WHERE reference_id = ? AND type = 'pnl_update' ORDER BY created_at DESC");
        $stmt->execute([$id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // --- HELPERS ---

    private function fetchCurrentPrice($type, $symbol) {
        if ($type == 'stock') {
            $data = $this->marketData->getStockPrice($symbol);
            return $data['price'];
        } elseif ($type == 'crypto') {
             $data = $this->marketData->getCryptoPrice($symbol); 
             return $data['price'];
        }
        return 100; // Mock default
    }

    private function getListingsByType($type) {
         if ($type == 'stock') {
            return [
                ['symbol' => 'AAPL', 'name' => 'Apple Inc.', 'type' => 'stock'],
                ['symbol' => 'GOOGL', 'name' => 'Alphabet Inc.', 'type' => 'stock'],
                ['symbol' => 'MSFT', 'name' => 'Microsoft Corporation', 'type' => 'stock'],
                ['symbol' => 'AMZN', 'name' => 'Amazon.com Inc.', 'type' => 'stock'],
                ['symbol' => 'TSLA', 'name' => 'Tesla Inc.', 'type' => 'stock'],
                ['symbol' => 'NVDA', 'name' => 'NVIDIA Corporation', 'type' => 'stock'],
                ['symbol' => 'META', 'name' => 'Meta Platforms Inc.', 'type' => 'stock'],
                ['symbol' => 'NFLX', 'name' => 'Netflix Inc.', 'type' => 'stock'],
                ['symbol' => 'AMD', 'name' => 'Advanced Micro Devices', 'type' => 'stock'],
                ['symbol' => 'INTC', 'name' => 'Intel Corporation', 'type' => 'stock'],
                ['symbol' => 'DIS', 'name' => 'Walt Disney Company', 'type' => 'stock'],
                ['symbol' => 'BA', 'name' => 'Boeing Company', 'type' => 'stock'],
                ['symbol' => 'JPM', 'name' => 'JPMorgan Chase & Co.', 'type' => 'stock'],
                ['symbol' => 'V', 'name' => 'Visa Inc.', 'type' => 'stock'],
                ['symbol' => 'MA', 'name' => 'Mastercard Inc.', 'type' => 'stock'],
            ];
        }
        if ($type == 'etf') {
            return [
                ['symbol' => 'SPY', 'name' => 'SPDR S&P 500 ETF Trust', 'type' => 'etf'],
                ['symbol' => 'QQQ', 'name' => 'Invesco QQQ Trust', 'type' => 'etf'],
                ['symbol' => 'VOO', 'name' => 'Vanguard S&P 500 ETF', 'type' => 'etf'],
                ['symbol' => 'VTI', 'name' => 'Vanguard Total Stock Market ETF', 'type' => 'etf'],
                ['symbol' => 'IWM', 'name' => 'iShares Russell 2000 ETF', 'type' => 'etf'],
                ['symbol' => 'DIA', 'name' => 'SPDR Dow Jones Industrial Average ETF', 'type' => 'etf'],
                ['symbol' => 'EEM', 'name' => 'iShares MSCI Emerging Markets ETF', 'type' => 'etf'],
                ['symbol' => 'GLD', 'name' => 'SPDR Gold Shares', 'type' => 'etf'],
                ['symbol' => 'TLT', 'name' => 'iShares 20+ Year Treasury Bond ETF', 'type' => 'etf'],
                ['symbol' => 'XLF', 'name' => 'Financial Select Sector SPDR Fund', 'type' => 'etf'],
            ];
        }
        if ($type == 'crypto') {
            return [
                 ['symbol' => 'bitcoin', 'api_id' => 'bitcoin', 'name' => 'Bitcoin', 'type' => 'crypto'],
                 ['symbol' => 'ethereum', 'api_id' => 'ethereum', 'name' => 'Ethereum', 'type' => 'crypto'],
                 ['symbol' => 'solana', 'api_id' => 'solana', 'name' => 'Solana', 'type' => 'crypto'],
            ];
        }
        if ($type == 'commodity') {
            return [
                 ['symbol' => 'gold', 'api_id' => 'pax-gold', 'name' => 'Gold (PAXG)', 'type' => 'commodity'],
                 ['symbol' => 'silver', 'api_id' => 'kinesis-silver', 'name' => 'Silver (KAG)', 'type' => 'commodity'],
            ];
        }
        if ($type == 'real_estate') {
             return [
                 ['symbol' => 'PROP-NY-01', 'name' => 'Manhattan Luxury Condo', 'purchase_price' => 1200000, 'min_invest' => 10000],
                 ['symbol' => 'PROP-MIA-02', 'name' => 'Miami Beach Villa', 'purchase_price' => 850000, 'min_invest' => 5000],
             ];
        }
        if ($type == 'fixed_income') {
             return [
                 ['symbol' => 'US-10Y', 'name' => 'US Treasury 10Y Bond', 'yield' => 4.5, 'type' => 'fixed_income', 'purchase_price' => 100],
                 ['symbol' => 'US-2Y', 'name' => 'US Treasury 2Y Bond', 'yield' => 4.9, 'type' => 'fixed_income', 'purchase_price' => 100],
             ];
        }
        return [];
    }

    private function recordTransaction($userId, $type, $amount, $description) {
        $stmt = $this->pdo->prepare("
            INSERT INTO transactions (user_id, type, amount, description)
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([$userId, $type, $amount, $description]);
    }
}