<?php

require_once __DIR__ . '/../utils/db.php';

class AutoInvestController {
    private $pdo;

    public function __construct() {
        $this->pdo = getPDO();
    }

    public function index() {
        // LIST plans
        $stmt = $this->pdo->query("SELECT * FROM special_plans WHERE status = 'active'");
        echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
    }

    public function getUserInvestments($userId) {
        $this->updateHourlyGrowth($userId); // Lazy update

        $stmt = $this->pdo->prepare("
            SELECT * FROM user_special_investments 
            WHERE user_id = ?
        ");
        $stmt->execute([$userId]);
        echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
    }

    public function invest($userId) {
        // Get JSON input
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!$data) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON input']);
            return;
        }
        
        $planId = $data['plan_id'] ?? null;
        $amount = $data['amount'] ?? null;
        
        if (!$planId || !$amount) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing plan_id or amount']);
            return;
        }
        
        // 1. Get Plan Details
        $stmt = $this->pdo->prepare("SELECT * FROM special_plans WHERE id = ?");
        $stmt->execute([$planId]);
        $plan = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$plan) {
            http_response_code(404);
            echo json_encode(['error' => 'Plan not found']);
            return;
        }

        // Validate amount
        if ($amount < $plan['min_amount'] || ($plan['max_amount'] && $amount > $plan['max_amount'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Amount outside plan limits']);
            return;
        }

        // 2. Check user balance
        $stmt = $this->pdo->prepare("SELECT funded_balance FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user || $user['funded_balance'] < $amount) {
            http_response_code(400);
            echo json_encode(['error' => 'Insufficient balance']);
            return;
        }

        // 3. Calculate expected return and hourly growth
        $totalROI = $amount * ($plan['roi_percentage'] / 100);
        $expectedReturn = $amount + $totalROI;
        $totalHours = $plan['duration_days'] * 24;
        $hourlyGrowth = $totalROI / $totalHours;

        // 4. Deduct balance
        $stmt = $this->pdo->prepare("UPDATE users SET funded_balance = funded_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $userId]);

        // 5. Create Investment
        $endDate = date('Y-m-d H:i:s', strtotime("+{$plan['duration_days']} days"));
        $stmt = $this->pdo->prepare("
            INSERT INTO user_special_investments 
            (user_id, plan_id, amount, end_date, status, accrued_profit)
            VALUES (?, ?, ?, ?, 'active', 0)
        ");
        
        $stmt->execute([$userId, $planId, $amount, $endDate]);

        echo json_encode([
            'success' => true, 
            'message' => 'Investment started successfully',
            'investment_id' => $this->pdo->lastInsertId(),
            'expected_return' => $expectedReturn
        ]);
    }

    private function updateHourlyGrowth($userId) {
        $stmt = $this->pdo->prepare("
            SELECT i.id, i.amount, i.accrued_profit, i.start_date, i.plan_id, 
                   p.roi_percentage, p.duration_days
            FROM user_special_investments i
            JOIN special_plans p ON i.plan_id = p.id
            WHERE i.user_id = ? AND i.status = 'active'
        ");
        $stmt->execute([$userId]);
        $investments = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($investments as $inv) {
            $startTime = strtotime($inv['start_date']);
            $now = time();
            $hoursPassed = ($now - $startTime) / 3600;
            
            // Calculate total expected profit
            $totalProfit = $inv['amount'] * ($inv['roi_percentage'] / 100);
            $totalHours = $inv['duration_days'] * 24;
            $hourlyRate = $totalProfit / $totalHours;
            
            // Calculate current accrued profit (capped at total)
            $newAccruedProfit = min($hourlyRate * $hoursPassed, $totalProfit);
            
            // Update if changed
            if (abs($newAccruedProfit - $inv['accrued_profit']) > 0.01) {
                $uStmt = $this->pdo->prepare("
                    UPDATE user_special_investments 
                    SET accrued_profit = ?
                    WHERE id = ?
                ");
                $uStmt->execute([$newAccruedProfit, $inv['id']]);
            }
        }
    }

    // ===== ADMIN METHODS =====
    
    public function getAllPlans() {
        // Get all plans including inactive (admin only)
        $stmt = $this->pdo->query("SELECT * FROM special_plans ORDER BY created_at DESC");
        echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
    }

    public function createPlan() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $required = ['name', 'description', 'min_amount', 'roi_percentage', 'duration_days', 'risk_level'];
        foreach ($required as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                return;
            }
        }

        $stmt = $this->pdo->prepare("
            INSERT INTO special_plans 
            (name, description, min_amount, max_amount, roi_percentage, duration_days, risk_level, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $data['name'],
            $data['description'],
            $data['min_amount'],
            $data['max_amount'] ?? null,
            $data['roi_percentage'],
            $data['duration_days'],
            $data['risk_level'],
            $data['status'] ?? 'active'
        ]);

        echo json_encode([
            'success' => true,
            'message' => 'Plan created successfully',
            'plan_id' => $this->pdo->lastInsertId()
        ]);
    }

    public function updatePlan($planId) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Build dynamic update query
        $fields = [];
        $values = [];
        
        $allowedFields = ['name', 'description', 'min_amount', 'max_amount', 'roi_percentage', 'duration_days', 'risk_level', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $values[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No fields to update']);
            return;
        }
        
        $values[] = $planId;
        $sql = "UPDATE special_plans SET " . implode(', ', $fields) . " WHERE id = ?";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($values);
        
        if ($stmt->rowCount() === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Plan not found']);
            return;
        }

        echo json_encode(['success' => true, 'message' => 'Plan updated successfully']);
    }

    public function deletePlan($planId) {
        // Check if plan has active investments
        $stmt = $this->pdo->prepare("
            SELECT COUNT(*) as count FROM user_special_investments 
            WHERE plan_id = ? AND status = 'active'
        ");
        $stmt->execute([$planId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result['count'] > 0) {
            http_response_code(400);
            echo json_encode([
                'error' => 'Cannot delete plan with active investments',
                'active_investments' => $result['count']
            ]);
            return;
        }

        // Soft delete by setting status to inactive
        $stmt = $this->pdo->prepare("UPDATE special_plans SET status = 'inactive' WHERE id = ?");
        $stmt->execute([$planId]);
        
        if ($stmt->rowCount() === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Plan not found']);
            return;
        }

        echo json_encode(['success' => true, 'message' => 'Plan deactivated successfully']);
    }
}
