<?php
require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../utils/jwt.php';
require_once __DIR__ . '/../utils/encryption.php';

class AuthController {
    private $pdo;
    public function __construct() {
        $this->pdo = getPDO();
    }

    public function register($data) {
        // Validate required fields
        if (!isset($data['email'], $data['password']) || !filter_var($data['email'], FILTER_VALIDATE_EMAIL) || strlen($data['password']) < 8) {
            return ['error' => 'Invalid email or password', 'code' => 400];
        }
        $stmt = $this->pdo->prepare('SELECT id FROM users WHERE email = ?');
        $stmt->execute([$data['email']]);
        if ($stmt->fetch()) {
            return ['error' => 'Email already registered', 'code' => 409];
        }
        $hash = password_hash($data['password'], PASSWORD_BCRYPT);
        $stmt = $this->pdo->prepare('INSERT INTO users (
            email, password_hash, account_type, investment_management, first_name, last_name, full_name, avatar_url, date_of_birth, country, ssn, nin, sin, tax_id, passport_number, address, city, state, zip_code, phone, id_type, id_number, selfie_url, referral_id, agreed_to_terms, funded_balance, demo_balance
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
        $stmt->execute([
            $data['email'],
            $hash,
            $data['accountType'] ?? null,
            $data['investmentManagement'] ?? null,
            $data['firstName'] ?? null,
            $data['lastName'] ?? null,
            $data['full_name'] ?? null,
            $data['avatar_url'] ?? null,
            $data['dateOfBirth'] ?? null,
            $data['country'] ?? null,
            encrypt_data($data['ssn'] ?? null),
            encrypt_data($data['nin'] ?? null),
            encrypt_data($data['sin'] ?? null),
            encrypt_data($data['taxId'] ?? null),
            encrypt_data($data['passportNumber'] ?? null),
            $data['address'] ?? null,
            $data['city'] ?? null,
            $data['state'] ?? null,
            $data['zipCode'] ?? null,
            $data['phone'] ?? null,
            $data['idType'] ?? null,
            $data['idNumber'] ?? null,
            $data['selfie_url'] ?? null,
            $data['referralId'] ?? null,
            !empty($data['agreeToTerms']) ? 1 : 0,
            00.00, // funded_balance
            00.00 // demo_balance
        ]);
        return ['success' => true];
    }

    public function login($email, $password) {
        error_log("=== LOGIN ATTEMPT START ===");
        error_log("Email: " . $email);
        error_log("Password length: " . strlen($password));
        
        $stmt = $this->pdo->prepare('SELECT * FROM users WHERE email = ?');
        $stmt->execute([$email]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            error_log("ERROR: User not found for email: " . $email);
            return ['error' => 'Invalid credentials', 'code' => 401];
        }
        
        error_log("User found - ID: " . $user['id'] . ", Email: " . $user['email']);
        error_log("Stored password hash: " . substr($user['password_hash'], 0, 20) . "...");
        
        $passwordValid = password_verify($password, $user['password_hash']);
        error_log("Password verification result: " . ($passwordValid ? 'VALID' : 'INVALID'));
        
        if (!$passwordValid) {
            error_log("ERROR: Password verification failed");
            return ['error' => 'Invalid credentials', 'code' => 401];
        }
        
        error_log("Creating JWT token...");
        $jwt = create_jwt($user);
        $refresh_token = bin2hex(random_bytes(64));
        $expires_at = date('Y-m-d H:i:s', time() + intval($_ENV['REFRESH_EXPIRE']));
        
        error_log("Storing refresh token...");
        $stmt = $this->pdo->prepare('INSERT INTO refresh_tokens (user_id, token, expires_at) VALUES (?, ?, ?)');
        $stmt->execute([$user['id'], $refresh_token, $expires_at]);
        
        error_log("=== LOGIN SUCCESS ===");
        return [
            'token' => $jwt,
            'refresh_token' => $refresh_token,
            'expires_in' => intval($_ENV['JWT_EXPIRE']),
            'user' => [
                'id' => $user['id'],
                'email' => $user['email'],
                'role' => $user['role'],
                'full_name' => $user['full_name'],
                'avatar_url' => $user['avatar_url'],
                'selfie_url' => $user['selfie_url'],
                'created_at' => $user['created_at'],
                'funded_balance' => $user['funded_balance'],
                'demo_balance' => $user['demo_balance'],
            ]
        ];
    }

    public function refresh($refresh_token) {
        $stmt = $this->pdo->prepare('SELECT * FROM refresh_tokens WHERE token = ? AND expires_at > NOW()');
        $stmt->execute([$refresh_token]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            return ['error' => 'Invalid or expired refresh token', 'code' => 401];
        }
        $stmt = $this->pdo->prepare('SELECT * FROM users WHERE id = ?');
        $stmt->execute([$row['user_id']]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$user) {
            return ['error' => 'User not found', 'code' => 404];
        }
        $jwt = create_jwt($user);
        return [
            'token' => $jwt,
            'expires_in' => intval($_ENV['JWT_EXPIRE'])
        ];
    }

    public function logout($refresh_token) {
        $stmt = $this->pdo->prepare('DELETE FROM refresh_tokens WHERE token = ?');
        $stmt->execute([$refresh_token]);
        return ['success' => true];
    }

    public function getProfile($user_id) {
        $stmt = $this->pdo->prepare('SELECT id, email, role, full_name, avatar_url, selfie_url, created_at, funded_balance, demo_balance, demo_first_used_at FROM users WHERE id = ?');
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$user) {
            return ['error' => 'User not found', 'code' => 404];
        }
        return ['user' => $user];
    }

    public function getDemoStatus($user_id) {
        $stmt = $this->pdo->prepare('SELECT demo_created_at, demo_notified FROM users WHERE id = ?');
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return ['error' => 'User not found', 'code' => 404];
        }
        
        return [
            'demo_created_at' => $user['demo_created_at'],
            'demo_notified' => (bool)$user['demo_notified']
        ];
    }

    public function updateDemoNotified($user_id) {
        $stmt = $this->pdo->prepare('UPDATE users SET demo_notified = TRUE WHERE id = ?');
        $stmt->execute([$user_id]);
        
        return ['success' => true, 'message' => 'Demo notification status updated'];
    }

    public function requestWithdrawal($user_id, $investment_id, $amount) {
        // Check if investment belongs to user and get manager permissions
        $stmt = $this->pdo->prepare('
            SELECT i.*, m.allow_withdrawals, m.withdrawal_message 
            FROM investments i 
            JOIN managers m ON i.manager_id = m.id 
            WHERE i.id = ? AND i.user_id = ? AND i.status = "active"
        ');
        $stmt->execute([$investment_id, $user_id]);
        $investment = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$investment) {
            return ['error' => 'Investment not found or not active', 'code' => 404];
        }

        // If demo account, enforce manager pause and demo trial window for demo-related withdrawals
        if (strtolower($investment['account_type']) !== 'funded') {
            // Manager pause check
            $pauseStmt = $this->pdo->prepare('SELECT demo_paused, expires_at FROM user_restrictions WHERE user_id = ?');
            $pauseStmt->execute([$user_id]);
            $restr = $pauseStmt->fetch(PDO::FETCH_ASSOC);
            $now = new DateTime();
            if ($restr && intval($restr['demo_paused']) === 1) {
                if (empty($restr['expires_at']) || (new DateTime($restr['expires_at'])) > $now) {
                    return [
                        'error' => 'Demo usage paused by manager. Please use a real account.',
                        'code' => 403
                    ];
                }
            }

            // 3-day trial enforcement since first demo use
            $trialStmt = $this->pdo->prepare('SELECT demo_first_used_at FROM users WHERE id = ?');
            $trialStmt->execute([$user_id]);
            $trialRow = $trialStmt->fetch(PDO::FETCH_ASSOC);
            if (!empty($trialRow['demo_first_used_at'])) {
                $first = new DateTime($trialRow['demo_first_used_at']);
                $diff = $first->diff($now);
                if ($diff->days >= 3) {
                    return [
                        'error' => 'Demo trial period exhausted. Please use a real account.',
                        'code' => 403
                    ];
                }
            }
        }
        
        if (!$investment['allow_withdrawals']) {
            return [
                'error' => 'Withdrawals not permitted', 
                'message' => $investment['withdrawal_message'] ?: 'This manager has disabled withdrawals for their investments.',
                'code' => 403
            ];
        }
        
        if ($amount > $investment['amount'] + $investment['pnl']) {
            return ['error' => 'Insufficient investment balance', 'code' => 400];
        }
        
        // Create withdrawal request
        $stmt = $this->pdo->prepare('INSERT INTO withdrawal_requests (investment_id, user_id, manager_id, amount) VALUES (?, ?, ?, ?)');
        $stmt->execute([$investment_id, $user_id, $investment['manager_id'], $amount]);
        
        return ['success' => true, 'request_id' => $this->pdo->lastInsertId()];
    }

    public function cancelInvestment($user_id, $investment_id) {
        // Check if investment belongs to user and get manager permissions
        $stmt = $this->pdo->prepare('
            SELECT i.*, m.allow_cancellations, m.cancellation_message 
            FROM investments i 
            JOIN managers m ON i.manager_id = m.id 
            WHERE i.id = ? AND i.user_id = ?
        ');
        $stmt->execute([$investment_id, $user_id]);
        $investment = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$investment) {
            return ['error' => 'Investment not found', 'code' => 404];
        }
        
        if (!$investment['allow_cancellations']) {
            return [
                'error' => 'Cancellations not permitted', 
                'message' => $investment['cancellation_message'] ?: 'This manager has disabled cancellations for their investments.',
                'code' => 403
            ];
        }
        
        if ($investment['status'] === 'cancelled') {
            return ['error' => 'Investment already cancelled', 'code' => 400];
        }
        
        $this->pdo->beginTransaction();
        
        try {
            // Update investment status
            $stmt = $this->pdo->prepare('UPDATE investments SET status = "cancelled" WHERE id = ?');
            $stmt->execute([$investment_id]);
            
            // Refund the investment amount to user balance
            $balanceField = strtolower($investment['account_type']) === 'funded' ? 'funded_balance' : 'demo_balance';
            $refundAmount = $investment['amount'] + max(0, $investment['pnl']); // Only refund positive PnL
            
            $stmt = $this->pdo->prepare("UPDATE users SET $balanceField = $balanceField + ? WHERE id = ?");
            $stmt->execute([$refundAmount, $user_id]);
            
            // Create transaction record for the refund
            $stmt = $this->pdo->prepare('INSERT INTO transactions (user_id, asset_id, type, amount, quantity, price, status) VALUES (?, 1, "refund", ?, 1, ?, "completed")');
            $stmt->execute([$user_id, $refundAmount, $refundAmount]);
            
            $this->pdo->commit();
            
            return ['success' => true, 'refund_amount' => $refundAmount];
            
        } catch (Exception $e) {
            $this->pdo->rollBack();
            return ['error' => 'Failed to cancel investment: ' . $e->getMessage(), 'code' => 500];
        }
    }

    // Signup with invitation token
    public function signupWithInvitation($data) {
        try {
            $full_name = $data['full_name'] ?? '';
            $email = $data['email'] ?? '';
            $password = $data['password'] ?? '';
            $invitation_token = $data['invitation_token'] ?? '';

            if (empty($full_name) || empty($email) || empty($password) || empty($invitation_token)) {
                return ['error' => 'All fields are required'];
            }

            // Validate email format
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                return ['error' => 'Invalid email format'];
            }

            // Check if email already exists
            $stmt = $this->pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            if ($stmt->fetchAll(PDO::FETCH_ASSOC)) {
                return ['error' => 'Email already registered'];
            }

            // Validate invitation token
            $stmt = $this->pdo->prepare("SELECT * FROM joint_account_invitations WHERE invitation_token = ? AND status = 'pending' AND expires_at > NOW()");
            $stmt->execute([$invitation_token]);
            $invitation = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (empty($invitation)) {
                return ['error' => 'Invalid or expired invitation'];
            }

            $invitation = $invitation[0];

            // Verify email matches invitation
            if ($invitation['invitee_email'] !== $email) {
                return ['error' => 'Email does not match invitation'];
            }

            // Start transaction
            $this->pdo->beginTransaction();

            try {
                // Create user account
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $this->pdo->prepare("INSERT INTO users (full_name, email, password, account_type, status, created_at) VALUES (?, ?, ?, 'individual', 'active', NOW())");
                $stmt->execute([$full_name, $email, $hashed_password]);
                $user_id = $this->pdo->lastInsertId();

                // Create user settings
                $stmt = $this->pdo->prepare("INSERT INTO user_settings (user_id, setting_key, setting_value) VALUES (?, 'email_notifications', 'enabled')");
                $stmt->execute([$user_id]);

                // Create default portfolio
                $stmt = $this->pdo->prepare("INSERT INTO portfolios (user_id, name, balance, created_at) VALUES (?, 'Main Portfolio', 0.00, NOW())");
                $stmt->execute([$user_id]);

                // Accept the invitation
                $joint_account_controller = new JointAccountController();
                $accept_result = $joint_account_controller->acceptInvitation($user_id, $invitation_token);

                if (!$accept_result['success']) {
                    throw new Exception($accept_result['error']);
                }

                // Generate JWT token
                $token = generate_jwt($user_id, $email);

                // Get user data
                $stmt = $this->pdo->prepare("SELECT id, full_name, email, account_type, status FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetchAll(PDO::FETCH_ASSOC)[0];

                $this->pdo->commit();

                return [
                    'success' => true,
                    'message' => 'Account created and invitation accepted successfully',
                    'token' => $token,
                    'user' => $user
                ];

            } catch (Exception $e) {
                $this->pdo->rollback();
                throw $e;
            }

        } catch (Exception $e) {
            return ['error' => 'Server error: ' . $e->getMessage()];
        }
    }
} 