<?php
require_once __DIR__ . '/../router.php';
require_once __DIR__ . '/../utils/db.php';
require_once __DIR__ . '/../utils/jwt.php';

$pdo = getPDO();

// GET: List active plans
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // If querying specific user investments
    if (isset($_GET['my_investments'])) {
        $user = validateJWT();
        if (!$user) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            exit;
        }
        $stmt = $pdo->prepare("
            SELECT i.*, p.name as plan_name, p.roi_percentage, p.duration_days 
            FROM user_special_investments i 
            JOIN special_plans p ON i.plan_id = p.id 
            WHERE i.user_id = ? 
            ORDER BY i.created_at DESC
        ");
        $stmt->execute([$user['userId']]);
        echo json_encode(['success' => true, 'investments' => $stmt->fetchAll(PDO::FETCH_ASSOC)]);
        exit;
    }

    // List available plans
    $stmt = $pdo->query("SELECT * FROM special_plans WHERE status = 'active' ORDER BY min_amount ASC");
    echo json_encode(['success' => true, 'plans' => $stmt->fetchAll(PDO::FETCH_ASSOC)]);
    exit;
}

// POST: Invest in a plan
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $user = validateJWT();
    if (!$user) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        exit;
    }

    $data = json_decode(file_get_contents('php://input'), true);
    $planId = $data['plan_id'] ?? null;
    $amount = $data['amount'] ?? 0;

    if (!$planId || !$amount) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing plan ID or amount']);
        exit;
    }

    // 1. Verify Plan
    $planStmt = $pdo->prepare("SELECT * FROM special_plans WHERE id = ? AND status = 'active'");
    $planStmt->execute([$planId]);
    $plan = $planStmt->fetch(PDO::FETCH_ASSOC);

    if (!$plan) {
        http_response_code(404);
        echo json_encode(['error' => 'Plan not found or inactive']);
        exit;
    }

    // 2. Verify Limits
    if ($amount < $plan['min_amount']) {
        http_response_code(400);
        echo json_encode(['error' => "Minimum investment is {$plan['min_amount']}"]);
        exit;
    }
    if ($plan['max_amount'] && $amount > $plan['max_amount']) {
        http_response_code(400);
        echo json_encode(['error' => "Maximum investment is {$plan['max_amount']}"]);
        exit;
    }

    // 3. Verify User Balance (Assuming 'funded_balance' in users table)
    $userStmt = $pdo->prepare("SELECT funded_balance FROM users WHERE id = ?");
    $userStmt->execute([$user['userId']]);
    $userBalance = $userStmt->fetchColumn();

    if ($userBalance < $amount) {
        http_response_code(400);
        echo json_encode(['error' => 'Insufficient funds']);
        exit;
    }

    try {
        $pdo->beginTransaction();

        // Deduct balance
        $deduct = $pdo->prepare("UPDATE users SET funded_balance = funded_balance - ? WHERE id = ?");
        $deduct->execute([$amount, $user['userId']]);

        // Create Investment
        $invest = $pdo->prepare("
            INSERT INTO user_special_investments (user_id, plan_id, amount, end_date) 
            VALUES (?, ?, ?, DATE_ADD(NOW(), INTERVAL ? DAY))
        ");
        $invest->execute([$user['userId'], $planId, $amount, $plan['duration_days']]);

        $pdo->commit();
        echo json_encode(['success' => true, 'message' => 'Investment successful']);
    } catch (Exception $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Transaction failed: ' . $e->getMessage()]);
    }
    exit;
}
