# Real API Integration Setup Guide

This guide explains how to set up real API integrations for fetching live market data in the Wealth Crypto Nexus application.

## Overview

The application now supports fetching real-time data from external APIs for:
- **Stocks**: Alpha Vantage API
- **Cryptocurrencies**: CoinGecko API  
- **Forex**: Alpha Vantage API

## API Configuration

### 1. Alpha Vantage API (Stocks & Forex)

**Get your free API key:**
1. Visit: https://www.alphavantage.co/support/#api-key
2. Sign up for a free account
3. Get your API key from the dashboard

**Free tier limits:**
- 5 API requests per minute
- 500 requests per day
- Real-time and historical data

### 2. CoinGecko API (Cryptocurrencies)

**Get your API key (optional):**
1. Visit: https://www.coingecko.com/en/api
2. Sign up for a free account
3. Get your API key (optional for basic usage)

**Free tier limits:**
- 10-50 calls per minute
- No API key required for basic usage
- Real-time cryptocurrency data

## Setup Instructions

### Step 1: Configure API Keys

Edit the file `backend/config/api-keys.php`:

```php
// Replace 'demo' with your actual Alpha Vantage API key
define('ALPHA_VANTAGE_API_KEY', 'YOUR_ALPHA_VANTAGE_API_KEY_HERE');

// Optional: Add your CoinGecko API key (leave empty for free tier)
define('COINGECKO_API_KEY', 'YOUR_COINGECKO_API_KEY_HERE');
```

### Step 2: Enable Real API Calls

In the same file, ensure real API calls are enabled:

```php
// Set to true to use real APIs, false to use fallback data only
define('ENABLE_REAL_API_CALLS', true);
```

### Step 3: Test the Integration

1. Start your backend server
2. Test the endpoints:
   - `/api/popular-stocks` - Should return real stock data
   - `/api/popular-crypto` - Should return real crypto data  
   - `/api/forex-pairs` - Should return real forex data

## API Endpoints

### Stocks API
- **Endpoint**: `/api/popular-stocks`
- **Source**: Alpha Vantage Global Quote API
- **Data**: Real-time stock prices, changes, volume
- **Rate Limit**: 5 requests/minute

### Cryptocurrency API
- **Endpoint**: `/api/popular-crypto`
- **Source**: CoinGecko Simple Price API
- **Data**: Real-time crypto prices, market cap, 24h changes
- **Rate Limit**: 10-50 requests/minute

### Forex API
- **Endpoint**: `/api/forex-pairs`
- **Source**: Alpha Vantage FX Daily API
- **Data**: Real-time forex rates, daily changes
- **Rate Limit**: 5 requests/minute

## Fallback System

The system includes a robust fallback mechanism:

1. **Primary**: Try to fetch from user's database (watchlist data)
2. **Secondary**: Fetch from real APIs (Alpha Vantage, CoinGecko)
3. **Tertiary**: Use static fallback data if APIs fail

## Error Handling

- API timeouts are set to 10 seconds
- Rate limiting delays prevent API abuse
- Comprehensive error logging
- Graceful fallback to static data

## Performance Considerations

- API calls include delays to respect rate limits
- Responses are cached for 5 minutes (configurable)
- Timeout settings prevent hanging requests
- User agent headers identify the application

## Troubleshooting

### Common Issues

1. **"API key invalid" errors**
   - Verify your API key is correct
   - Check if you've exceeded rate limits
   - Ensure your account is active

2. **"No data returned" errors**
   - Check your internet connection
   - Verify API endpoints are accessible
   - Check server error logs

3. **Rate limit exceeded**
   - Wait for the rate limit window to reset
   - Consider upgrading to a paid API plan
   - Reduce the number of concurrent requests

### Debug Mode

Enable debug logging by checking the server error logs for API-related messages.

## Alternative APIs

If you prefer different data sources, you can modify the API functions in:
- `backend/api/popular-stocks.php` - `fetchRealStockData()`
- `backend/api/popular-crypto.php` - `fetchRealCryptoData()`
- `backend/api/forex-pairs.php` - `fetchRealForexData()`

### Alternative Services

**Stocks:**
- Yahoo Finance (free, no API key)
- IEX Cloud (free tier available)
- Polygon.io (free tier available)
- Twelve Data (free tier available)

**Cryptocurrencies:**
- CoinMarketCap API
- CryptoCompare API
- Binance API

**Forex:**
- Fixer.io
- CurrencyLayer
- ExchangeRate-API

## Security Notes

- Never commit API keys to version control
- Use environment variables for production
- Rotate API keys regularly
- Monitor API usage for unusual activity

## Support

For issues with specific APIs:
- Alpha Vantage: https://www.alphavantage.co/support/
- CoinGecko: https://www.coingecko.com/en/api/documentation

For application-specific issues, check the server logs and ensure all configuration files are properly set up.
